/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TextObject.C
 *    Adapted from http://www.ogre3d.org/tikiwiki/MovableText&structure=Cookbook
 *
 * Original file: MovableText.cpp
 *
 * description: This create create a billboarding object that display a text.
 *
 * @author  2003 by cTh see gavocanov@rambler.ru
 * @update  2006 by barraq see nospam@barraquand.com
 */

#include <OGRE/Ogre.h>
#if (OGRE_VERSION >= 0x010900)
#include <OGRE/Overlay/OgreFontManager.h>
#else
#include <OGRE/OgreFontManager.h>
#endif
#include <visualization/3d/TextObject.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#define POS_TEX_BINDING    0
#define COLOUR_BINDING     1


TextObject::TextObject(const Ogre::String &caption,
                       Ogre::SceneManager* sceneManager,
                       Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mFontName("FreeSans"),
	mType("TextObject"),
	mCaption(caption),
	mHorizontalAlignment(H_LEFT),
	mVerticalAlignment(V_BELOW),
	mColor(Ogre::ColourValue::White),
	mCharHeight(1.0),
	mSpaceWidth(0),
	mUpdateColors(true),
	mOnTop(false),
	mTimeUntilNextToggle(0),
	mGlobalTranslation(0.0),
	mLocalTranslation(0.0), mCam(NULL),
	mWin(NULL),
	mFont(NULL)
{
	mName = "TextObject" + toString(this);
	mRenderOp.vertexData = NULL;
	this->setFontName(mFontName);
	this->_setupGeometry();
	mNode->attachObject(this);
}

TextObject::TextObject(const Ogre::String &caption,
                       const Ogre::String &fontName,
                       Ogre::SceneManager* sceneManager,
                       Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mFontName(fontName),
	mType("TextObject"),
	mCaption(caption),
	mHorizontalAlignment(H_LEFT),
	mVerticalAlignment(V_BELOW),
	mColor(Ogre::ColourValue::White),
	mCharHeight(1.0),
	mSpaceWidth(0),
	mUpdateColors(true),
	mOnTop(false),
	mTimeUntilNextToggle(0),
	mGlobalTranslation(0.0),
	mLocalTranslation(0.0), mCam(NULL),
	mWin(NULL),
	mFont(NULL)
{
	mName = "TextObject" + toString(this);
	mRenderOp.vertexData = NULL;
	this->setFontName(mFontName);
	this->_setupGeometry();
	mNode->attachObject(this);
}

TextObject::~TextObject()
{
	mNode->detachObject(this);
	if (mRenderOp.vertexData)
		delete mRenderOp.vertexData;
	// May cause crashing... check this and comment if it does
	if (!mMaterial.isNull())
		Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());
}

void TextObject::setFontName(const Ogre::String &fontName)
{
	if ((Ogre::MaterialManager::getSingletonPtr()->resourceExists(mName
			+ "Material")))
	{
		Ogre::MaterialManager::getSingleton().remove(mName + "Material");
	}

	if (mFontName != fontName || mMaterial.isNull() || !mFont)
	{
		mFontName = fontName;
		mFont     = (Ogre::Font *) Ogre::FontManager::getSingleton().getByName(mFontName).getPointer();
		if (!mFont)
			throw Ogre::Exception(Ogre::Exception::ERR_ITEM_NOT_FOUND,
					"Could not find font " + fontName,
					"TextObject::setFontName");

		mFont->load();
		if (!mMaterial.isNull())
		{
			Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());
			mMaterial.setNull();
		}

		mMaterial = mFont->getMaterial()->clone(mName + "Material");
		if (!mMaterial->isLoaded())
			mMaterial->load();

		mMaterial->setDepthCheckEnabled(!mOnTop);
		mMaterial->setDepthBias(1.0, 1.0);
		mMaterial->setDepthWriteEnabled(mOnTop);
		mMaterial->setLightingEnabled(false);
		mNeedUpdate = true;
	}
}

void TextObject::setCaption(const Ogre::String &caption)
{
	if (caption != mCaption)
	{
		mCaption = caption;
		mNeedUpdate = true;
	}
}

void TextObject::setColor(const Ogre::ColourValue &color)
{
	if (color != mColor)
	{
		mColor = color;
		mUpdateColors = true;
	}
}

void TextObject::setVisible(bool visible, bool cascade)
{
	VisualizationObject::setVisible(visible, cascade);
}

void TextObject::setCharacterHeight(Ogre::Real height)
{
	if (height != mCharHeight)
	{
		mCharHeight = height;
		mNeedUpdate = true;
	}
}

void TextObject::setSpaceWidth(Ogre::Real width)
{
	if (width != mSpaceWidth)
	{
		mSpaceWidth = width;
		mNeedUpdate = true;
	}
}

void TextObject::setTextAlignment(
		const HorizontalAlignment& horizontalAlignment,
		const VerticalAlignment& verticalAlignment)
{
	if (mHorizontalAlignment != horizontalAlignment)
	{
		mHorizontalAlignment = horizontalAlignment;
		mNeedUpdate = true;
	}
	if (mVerticalAlignment != verticalAlignment)
	{
		mVerticalAlignment = verticalAlignment;
		mNeedUpdate = true;
	}
}

void TextObject::setGlobalTranslation(Ogre::Vector3 trans)
{
	mGlobalTranslation = trans;
}

void TextObject::setLocalTranslation(Ogre::Vector3 trans)
{
	mLocalTranslation = trans;
}

void TextObject::showOnTop(bool show)
{
	if (mOnTop != show && !mMaterial.isNull())
	{
		mOnTop = show;
		mMaterial->setDepthBias(1.0, 1.0);
		mMaterial->setDepthCheckEnabled(!mOnTop);
		mMaterial->setDepthWriteEnabled(mOnTop);
	}
}

void TextObject::_setupGeometry()
{
	assert(mFont);
	assert(!mMaterial.isNull());

	Ogre::String caption = mCaption;
	if (caption.empty())
		caption=" "; // we need at least one char to avoid an empty vertex buffer

	unsigned int vertexCount = static_cast<unsigned int> (caption.size() * 6);

	if (mRenderOp.vertexData)
	{
		delete mRenderOp.vertexData;
		mRenderOp.vertexData = NULL;
		mUpdateColors = true;
	}


	if (!mRenderOp.vertexData)
		mRenderOp.vertexData = new Ogre::VertexData();

	mRenderOp.indexData = 0;
	mRenderOp.vertexData->vertexStart = 0;
	mRenderOp.vertexData->vertexCount = vertexCount;
	mRenderOp.operationType = Ogre::RenderOperation::OT_TRIANGLE_LIST;
	mRenderOp.useIndexes = false;

	Ogre::VertexDeclaration *decl = mRenderOp.vertexData->vertexDeclaration;
	Ogre::VertexBufferBinding *bind = mRenderOp.vertexData->vertexBufferBinding;
	size_t offset = 0;

	// create/bind positions/tex.ccord. buffer
	if (!decl->findElementBySemantic(Ogre::VES_POSITION))
		decl->addElement(POS_TEX_BINDING, offset, Ogre::VET_FLOAT3, Ogre::VES_POSITION);

	offset += Ogre::VertexElement::getTypeSize(Ogre::VET_FLOAT3);

	if (!decl->findElementBySemantic(Ogre::VES_TEXTURE_COORDINATES))
		decl->addElement(POS_TEX_BINDING, offset, Ogre::VET_FLOAT2,
		                 Ogre::VES_TEXTURE_COORDINATES, 0);

	Ogre::HardwareVertexBufferSharedPtr ptbuf =
			Ogre::HardwareBufferManager::getSingleton().createVertexBuffer(
					decl->getVertexSize(POS_TEX_BINDING),
					mRenderOp.vertexData->vertexCount,
					Ogre::HardwareBuffer::HBU_DYNAMIC_WRITE_ONLY);
	bind->setBinding(POS_TEX_BINDING, ptbuf);

	// Colours - store these in a separate buffer because they change less often
	if (!decl->findElementBySemantic(Ogre::VES_DIFFUSE))
		decl->addElement(COLOUR_BINDING, 0, Ogre::VET_COLOUR, Ogre::VES_DIFFUSE);

	Ogre::HardwareVertexBufferSharedPtr cbuf =
			Ogre::HardwareBufferManager::getSingleton().createVertexBuffer(
					decl->getVertexSize(COLOUR_BINDING),
					mRenderOp.vertexData->vertexCount,
					Ogre::HardwareBuffer::HBU_DYNAMIC_WRITE_ONLY);
	bind->setBinding(COLOUR_BINDING, cbuf);

	float *pPCBuff = static_cast<float*>(ptbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));

	Ogre::Real spaceWidth = mSpaceWidth;
	// Derive space width from a capital A
	if (spaceWidth == 0)
		spaceWidth = mFont->getGlyphAspectRatio('A') * mCharHeight * 2.0;

	float totalHeight = mCharHeight;
	float totalWidth = 0.0f;
	float currentWidth = 0.0f;
	std::string::iterator i = caption.begin();
	std::string::iterator iend = caption.end();
	for ( ; i != iend; ++i )
	{
		if (*i == '\n')
		{
			totalHeight += mCharHeight + 0.01;
			if ( currentWidth > totalWidth )
				totalWidth = currentWidth;
			currentWidth = 0.0f;
		}
		else
			currentWidth += mFont->getGlyphAspectRatio(*i) * mCharHeight * 2.0;
	}

	if ( currentWidth > totalWidth )
		totalWidth = currentWidth;

	float top = 0.0f;
	switch (mVerticalAlignment)
	{
		case V_ABOVE:
			top = totalHeight * 2;
			break;
		case V_CENTER:
			top = 0.5 * totalHeight * 2;
			break;
		case V_BELOW:
			top = 0.0f;
			break;
	}

	float startingLeft = 0.0f;
	switch (mHorizontalAlignment)
	{
		case H_LEFT:
			startingLeft = 0.0f;
			break;
		case H_CENTER:
			startingLeft = -totalWidth / 2.0f;
			break;
	}

	float left = startingLeft;

	bool newLine = true;
	Ogre::Real len = 0.0f;
	// for calculation of AABB
	Ogre::Vector3 min(9999999.0f), max(-9999999.0f), currPos(0.0f);
	Ogre::Real maxSquaredRadius = -99999999.0f;
	float largestWidth = 0.0f;
	for (i = caption.begin(); i != iend; ++i)
	{
		if (newLine)
		{
			len = 0.0f;
			for (Ogre::String::iterator j = i; j != iend && *j != '\n'; j++)
			{
				if (*j == ' ')
					len += spaceWidth;
				else
					len += mFont->getGlyphAspectRatio(*j) * mCharHeight * 2.0;
			}
			newLine = false;
		}

		if (*i == '\n')
		{
			left = startingLeft;
			top -= mCharHeight * 2.0;
			newLine = true;
			continue;
		}

		if (*i == ' ')
		{
			// Just leave a gap, no tris
			left += spaceWidth;
			// Also reduce tri count
			mRenderOp.vertexData->vertexCount -= 6;
			continue;
		}

		Ogre::Real horiz_height = mFont->getGlyphAspectRatio(*i);
		Ogre::Real u1, u2, v1, v2;
		Ogre::Font::UVRect utmp;
		utmp = mFont->getGlyphTexCoords(*i);
		u1 = utmp.left;
		u2 = utmp.right;
		v1 = utmp.top;
		v2 = utmp.bottom;

		// each vert is (x, y, z, u, v)
		//----------------------------------------------------------- ---------
		// First tri
		//
		// Upper left
		currPos = Ogre::Vector3(left, top, 0.0);
		*pPCBuff++ = currPos.x;
		*pPCBuff++ = currPos.y;
		*pPCBuff++ = currPos.z;
		*pPCBuff++ = u1;
		*pPCBuff++ = v1;

		// Deal with bounds
		min.makeFloor(currPos);
		max.makeCeil(currPos);
		maxSquaredRadius = std::max(maxSquaredRadius, currPos.squaredLength());

		top -= mCharHeight * 2.0;

		// Bottom left
		currPos = Ogre::Vector3(left, top, 0.0);
		*pPCBuff++ = currPos.x;
		*pPCBuff++ = currPos.y;
		*pPCBuff++ = currPos.z;
		*pPCBuff++ = u1;
		*pPCBuff++ = v2;

		// Deal with bounds
		min.makeFloor(currPos);
		max.makeCeil(currPos);
		maxSquaredRadius = std::max(maxSquaredRadius, currPos.squaredLength());

		top += mCharHeight * 2.0;
		left += horiz_height * mCharHeight * 2.0;

		// Top right
		currPos = Ogre::Vector3(left, top, 0.0);
		*pPCBuff++ = currPos.x;
		*pPCBuff++ = currPos.y;
		*pPCBuff++ = currPos.z;
		*pPCBuff++ = u2;
		*pPCBuff++ = v1;
		//---------------------------------------------------------------------

		// Deal with bounds
		min.makeFloor(currPos);
		max.makeCeil(currPos);
		maxSquaredRadius = std::max(maxSquaredRadius, currPos.squaredLength());

		//---------------------------------------------------------------------
		// Second tri
		//
		// Top right (again)
		currPos = Ogre::Vector3(left, top, 0.0);
		*pPCBuff++ = currPos.x;
		*pPCBuff++ = currPos.y;
		*pPCBuff++ = currPos.z;
		*pPCBuff++ = u2;
		*pPCBuff++ = v1;

		min.makeFloor(currPos);
		max.makeCeil(currPos);
		maxSquaredRadius = std::max(maxSquaredRadius, currPos.squaredLength());

		top -= mCharHeight * 2.0;
		left -= horiz_height * mCharHeight * 2.0;

		// Bottom left (again)
		currPos = Ogre::Vector3(left, top, 0.0);
		*pPCBuff++ = currPos.x;
		*pPCBuff++ = currPos.y;
		*pPCBuff++ = currPos.z;
		*pPCBuff++ = u1;
		*pPCBuff++ = v2;

		min.makeFloor(currPos);
		max.makeCeil(currPos);
		maxSquaredRadius = std::max(maxSquaredRadius, currPos.squaredLength());

		left += horiz_height * mCharHeight * 2.0;

		// Bottom right
		currPos = Ogre::Vector3(left, top, 0.0);
		*pPCBuff++ = currPos.x;
		*pPCBuff++ = currPos.y;
		*pPCBuff++ = currPos.z;
		*pPCBuff++ = u2;
		*pPCBuff++ = v2;
		//---------------------------------------------------------------------
		min.makeFloor(currPos);
		max.makeCeil(currPos);
		maxSquaredRadius = std::max(maxSquaredRadius, currPos.squaredLength());

		// Go back up with top
		top += mCharHeight * 2.0;

		float currentWidth = (left + 1) / 2 - 0;
		if (currentWidth > largestWidth)
			largestWidth = currentWidth;
	}

	// Unlock vertex buffer
	ptbuf->unlock();

	//for empty strings or only white space the min and max will not be set properly
	if(min.x <= max.x)
	{
		// update AABB/Sphere radius
		mAABB = Ogre::AxisAlignedBox(min, max);
		mRadius = Ogre::Math::Sqrt(maxSquaredRadius);
	}
	else
	{
		// update AABB/Sphere radius
		mAABB = Ogre::AxisAlignedBox(Ogre::Vector3(0.f), Ogre::Vector3(0.f));
		mRadius = Ogre::Math::Sqrt(0.);
	}

	if (mUpdateColors)
		this->_updateColors();

	mNeedUpdate = false;
}

void TextObject::_updateColors(void)
{
	assert(mFont);
	assert(!mMaterial.isNull());

	// Convert to system-specific
	Ogre::RGBA color;
	Ogre::Root::getSingleton().convertColourValue(mColor, &color);
	Ogre::HardwareVertexBufferSharedPtr
			vbuf = mRenderOp.vertexData->vertexBufferBinding->getBuffer(
					COLOUR_BINDING);
	Ogre::RGBA *pDest = static_cast<Ogre::RGBA*> (vbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));
	for (int i = 0; i < (int) mRenderOp.vertexData->vertexCount; ++i)
		*pDest++ = color;
	vbuf->unlock();
	mUpdateColors = false;
}

const Ogre::Quaternion& TextObject::getWorldOrientation(void) const
{
	assert(mCam);
	return const_cast<Ogre::Quaternion&> (mCam->getDerivedOrientation());
}

// Add to build on Shoggoth:
void TextObject::visitRenderables(Ogre::Renderable::Visitor* visitor,
		bool debugRenderables)
{
}

const Ogre::Vector3& TextObject::getWorldPosition(void) const
{
	assert(mParentNode);
	return mParentNode->_getDerivedPosition();
}

void TextObject::getWorldTransforms(Ogre::Matrix4 *xform) const
{
	if (this->isVisible() && mCam)
	{
		Ogre::Matrix3 rot3x3, scale3x3 = Ogre::Matrix3::IDENTITY;

		// store rotation in a matrix
		mCam->getDerivedOrientation().ToRotationMatrix(rot3x3);

		// parent node position
		Ogre::Vector3 ppos = mParentNode->_getDerivedPosition() + Ogre::Vector3::UNIT_Y
				* mGlobalTranslation;
		ppos += rot3x3 * mLocalTranslation;

		// apply scale
		scale3x3[0][0] = mParentNode->_getDerivedScale().x / 2;
		scale3x3[1][1] = mParentNode->_getDerivedScale().y / 2;
		scale3x3[2][2] = mParentNode->_getDerivedScale().z / 2;

		// apply all transforms to xform
		*xform = (rot3x3 * scale3x3);
		xform->setTrans(ppos);
	}
}

void TextObject::getRenderOperation(Ogre::RenderOperation &op)
{
	if (this->isVisible())
	{
		if (mNeedUpdate)
			this->_setupGeometry();
		if (mUpdateColors)
			this->_updateColors();
		op = mRenderOp;
	}
}

void TextObject::_notifyCurrentCamera(Ogre::Camera *cam)
{
	mCam = cam;
}

void TextObject::_updateRenderQueue(Ogre::RenderQueue* queue)
{
	if (this->isVisible())
	{
		if (mNeedUpdate)
			this->_setupGeometry();
		if (mUpdateColors)
			this->_updateColors();

		queue->addRenderable(this, mRenderQueueID,
				OGRE_RENDERABLE_DEFAULT_PRIORITY);
		//queue->addRenderable(this, mRenderQueueID, RENDER_QUEUE_SKIES_LATE);
	}
}

///////////////////////////////////////////////////////////////////////////////

}
