/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonTool.C
 *    Implementation of PolygonTool.
 *
 * @author Tim Langner
 * @date   2012/10/05
 */

#include <QMouseEvent>

#include <visualization/3d/PolygonTool.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void PolygonTool::onMouseReleased(QMouseEvent* e)
{
	if (e->button() == Qt::LeftButton)
	{
		auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
		if (!i.first.empty())
		{
			mSelectedObject = i.first;
			mPolygon.push_back(Point2f(i.second.x, i.second.y));
			mPolygonObject->setPolygon(mPolygon);
		}
	}
	else if (e->button() == Qt::RightButton)
	{
		if (mPolygon.size() < 3)
			MIRA_THROW(XInvalidConfig, "Polygon must at least have 3 points");
		mSelectedObject = "";
		boost::geometry::correct(mPolygon);
		onNewPolygon(mPolygon);
		mPolygon = Polygon2f();
	}
}

/// called when the user has activated the tool.
void PolygonTool::activate()
{
	mPolygonObject->setVisible(true);
}

/// called when the user has deactivated the tool by selecting another one.
void PolygonTool::deactivate()
{
	mPolygonObject->setVisible(false);
}

std::pair<std::string, Ogre::Vector3> PolygonTool::getIntersection(float x, float y,
                                                                   const std::string& mask)
{
	return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject")
                        : SceneQueryTool::getIntersection(x, y, mask, true);
}

void PolygonTool::setupScene(IVisualization3DSite* site)
{
	SceneQueryTool::setupScene(site);
	mPolygonObject.reset(new PolygonObject<float>(site->getSceneManager()));
	mPolygonObject->setVisible(false);
	mSelectedObject = "";
}

void PolygonTool::setPolygon(const Polygon2f& polygon)
{
	mPolygon = polygon;
	if (mPolygonObject)
		mPolygonObject->setPolygon(mPolygon);
}

///////////////////////////////////////////////////////////////////////////////

}
