/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTextView.C
 *    Implementation of Default text visualizations.
 *
 * @author Erik Einhorn
 * @date   2011/01/15
 */

#include <views/VisualizationTextView.h>

#include <QBasicTimer>
#include <QDragMoveEvent>
#include <QScrollBar>
#include <QVBoxLayout>

#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>

#include <views/VisualizationView.h>
#include <views/VisualizationControlPage.h>

#include <visualization/VisualizationTextBasic.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class DefaultTextVisualization : public VisualizationTextBasic<void>
{
MIRA_OBJECT(DefaultTextVisualization)

public:

	typedef VisualizationTextBasic<void> Base;

	DefaultTextVisualization() :
		Base("Channel"), mShowROProperties(true), mStringMapAsObject(true)
	{}

	template <typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Base);
		r.property("ShowReadOnlyProperties", mShowROProperties,
		           setterNotify(mShowROProperties, &DefaultTextVisualization::redraw, this),
		           "Show objects' read-only properties", true);
		r.property("StringMapAsObject", mStringMapAsObject,
		           setterNotify(mStringMapAsObject, &DefaultTextVisualization::redraw, this),
		           "Serialize any map<string, T> to json object "
		           "(enables simple access to map items by key)", true);
	}

	virtual void dataChanged(ChannelRead<void> data)
	{
		this->mItem->setStamped(data->timestamp, data->frameID, data->sequenceID);

		try
		{
			json::Value v;
			try {
				JSONSerializer js(mShowROProperties,
				                  this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
				                                           : JSONSerializer::STANDARD);
				data.readJSON(v, js);
			}
			catch(XNotImplemented&)
			{
				data.readJSON(v);
			}
			mLastValue.reset(v);
			mItem->setText(json::write(v,true,mPrecision));
		}
		catch(...)
		{
			int typeID = MIRA_FW.getChannelManager().getTypeId(Base::mChannel.getID());
			if (typeID < 0)
				this->mItem->setText("Untyped channel, conversion to json format (from metadata) not possible");
			else
				this->mItem->setText("Conversion to json format not supported by channel type");
		}
	}

	virtual void setupScene(IVisualizationTextItem* item)
	{
		item->registerTextEditRequestCallback(boost::bind(&DefaultTextVisualization::onEdit, this));
		item->enableEdit(true);
	}

	virtual void redraw()
	{
		try {
			if (mChannel.isValid())
				dataChanged(this->mChannel.getChannel().read());
		} catch (...) {}
	}

protected:
	bool mShowROProperties;
	bool mStringMapAsObject;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Text view that may contain different text visualization objects which
 * show channel content as text.
 */
class VisualizationTextView : public VisualizationViewTransformable, public IVisualizationTextSite
{
MIRA_META_OBJECT(VisualizationTextView,
                 ("Name", "Text View")
                 ("Description", "View for text visualizations")
                 ("Category"   , "Visualization")
                 ("Visualization", "mira::VisualizationText"))

public:
	VisualizationTextView();

	virtual ~VisualizationTextView()
	{
		destroyVisualizations();
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		VisualizationViewTransformable::reflect(r);
		r.property("Precision", mPrecision,
		           setter(&VisualizationTextView::setPrecision, this),
		           "Precision for text output of float numbers.", 3,
		           PropertyHints::minimum(0));
	}

public: // implementation of VisualizationView

	virtual void addVisualization(Visualization* vis);

	virtual void moveUpVisualization(Visualization* vis);
	virtual void moveDownVisualization(Visualization* vis);

	virtual const Class& supportedVisualizationClass() const;

	virtual const Class& defaultVisualizationClass() const;

protected:

	virtual void update(Duration dt);


protected:

	virtual QWidget* createVisualizationPart();

public: // implementation of IVisualizationTextSite

	virtual IVisualizationTextItem* addItem();
	virtual void destroyItem(IVisualizationTextItem* item);

	virtual const std::string& getFixedFrame() const {
		return mFixedFrame.getID();
	}

	virtual const std::string& getCameraFrame() const {
		return mCameraFrame.getID();
	}

private:
	void setPrecision(uint precision);
	uint mPrecision;

private:

	class UI;
	UI* ui;

private:
	friend class UI;
	void setupVisualizations();
};

///////////////////////////////////////////////////////////////////////////////

VisualizationTextItem::VisualizationTextItem(QTableWidget* t, int row) :
	QObject(t),
	table(t)
{
	nameItem = new QTableWidgetItem();
	nameItem->setTextAlignment(Qt::AlignLeft | Qt::AlignTop);
	table->setItem(row, 0, nameItem);
	table->setRowHeight(row, 50);

	textItem = new QPlainTextEdit();
	textItem->setReadOnly(true);
	assert(textItem->document()!=NULL);
	textItem->document()->setDocumentMargin(0);

	textItem->setFrameShape(QFrame::Panel);
	//textItem->setFrameStyle(0);
	textItem->setLineWidth(0);
	table->setCellWidget(row,1, textItem);

	timestampItem = new QTableWidgetItem();
	timestampItem->setTextAlignment(Qt::AlignLeft | Qt::AlignTop);
	table->setItem(row, 2, timestampItem);

	sequenceidItem = new QTableWidgetItem();
	sequenceidItem->setTextAlignment(Qt::AlignLeft | Qt::AlignTop);
	table->setItem(row, 3, sequenceidItem);

	frameidItem = new QTableWidgetItem();
	frameidItem->setTextAlignment(Qt::AlignLeft | Qt::AlignTop);
	table->setItem(row, 4, frameidItem);

	setItem = new QWidget();
	setButton = new QPushButton("Set",setItem);
	connect(setButton, SIGNAL(clicked()), this, SLOT(onSetClicked()));
	QVBoxLayout* layout = new QVBoxLayout();
	layout->addWidget(setButton);
	layout->addStretch();
	layout->setMargin(0);
	layout->setContentsMargins(0,0,0,0);
	setItem->setLayout(layout);
	setItem->setEnabled(false);
	table->setCellWidget(row, 5, setItem);
}

void VisualizationTextItem::onSetClicked()
{
	if (mEditRequestCb)
		mEditRequestCb();
}

void VisualizationTextItem::setName(const std::string& name)
{
	nameItem->setText(name.c_str());
}

void VisualizationTextItem::setText(const std::string& text)
{
	//int x = textItem->horizontalScrollBar()->value();
	//int y = textItem->verticalScrollBar()->value();

	textItem->blockSignals(true);
	textItem->setPlainText(text.c_str());
	textItem->blockSignals(false);

	//textItem->horizontalScrollBar()->setValue(x);
	//textItem->verticalScrollBar()->setValue(y);
}

void VisualizationTextItem::setStamped(const Time& timestamp, const std::string& frameID,
                                       uint32 sequenceID)
{
	timestampItem->setText(toString(timestamp.toLocal()).c_str());
	sequenceidItem->setText(toString(sequenceID).c_str());
	frameidItem->setText(frameID.c_str());
}

int VisualizationTextItem::row() const
{
	assert(nameItem!=NULL);
	return nameItem->row();
}

void VisualizationTextItem::enableEdit(bool enable)
{
	setItem->setEnabled(enable);
}

void VisualizationTextItem::registerTextEditRequestCallback(boost::function<void ()> fn)
{
	mEditRequestCb = fn;
}

void VisualizationTextItem::setSize(const Size2i& size)
{
	table->setRowHeight(row(), size.height());
	int w = std::max(table->columnWidth(1), size.width());
	table->setColumnWidth(1, w);
}

Size2i VisualizationTextItem::getSize() const
{
	int w = table->columnWidth(1);
	int h = table->rowHeight(row());
	return Size2i(w, h);
}

///////////////////////////////////////////////////////////////////////////////

class VisualizationTextView::UI : public QTableWidget
{
public:

	UI(VisualizationTextView* parent) : QTableWidget(parent), vis(parent)
	{
		setColumnCount(6);
		setAcceptDrops(true);
		setSelectionMode(QAbstractItemView::NoSelection);
		setHorizontalHeaderLabels(QStringList() << "Name" << "Value" << "Timestamp" << "SequenceID" << "FrameID" << "Set");
		//setAlternatingRowColors(true);
		setColumnWidth(1, 200);
		vis->startUpdateTimer();
	}

	virtual ~UI()
	{
	}

protected:

	virtual void dragEnterEvent(QDragEnterEvent *event)	{
		setAcceptDrops(true);
		vis->dragEnterEventImpl(event);
	}

	virtual void dropEvent(QDropEvent *event) {
		vis->dropEventImpl(event, this);
	}

	// we need to overwrite this to prevent the QGraphicsView from disturbing our drag n drop
	virtual void dragMoveEvent(QDragMoveEvent* event) {
		event->accept();
	}

	VisualizationTextView* vis;

};

VisualizationTextView::VisualizationTextView() : mPrecision(3), ui(NULL)
{
	setUpdateInterval(40);
}


void VisualizationTextView::setupVisualizations()
{
	// initialize all existing visualizations
	foreach(Visualization* vis, getVisualizations())
		vis->init(this);
}

void VisualizationTextView::addVisualization(Visualization* vis)
{
	VisualizationText* v = dynamic_cast<VisualizationText*>(vis);
	assert(v!=NULL);
	v->init(this);
	v->setPrecision(mPrecision);
	VisualizationView::addVisualization(vis);
}

void VisualizationTextView::moveUpVisualization(Visualization* vis)
{
	VisualizationViewTransformable::moveUpVisualization(vis);

	VisualizationText* v = dynamic_cast<VisualizationText*>(vis);
	assert(v!=NULL);
	IVisualizationTextItem* iv = v->getItem();
	VisualizationTextItem* item = dynamic_cast<VisualizationTextItem*>(iv);
	assert(item!=NULL);
	int row = item->row();

	if (row < 1)
		return;

	// insert at row-1  = before previous row -> row moves to row+1
	ui->insertRow(row-1);

	Size2i size = item->getSize();

	// move contents from row+1 to row-1
	ui->setItem(row-1, 0, ui->takeItem(row+1, 0));
	ui->setCellWidget(row-1, 1, ui->cellWidget(row+1, 1));
	ui->setItem(row-1, 2, ui->takeItem(row+1, 2));
	ui->setItem(row-1, 3, ui->takeItem(row+1, 3));
	ui->setItem(row-1, 4, ui->takeItem(row+1, 4));
	ui->setCellWidget(row-1, 5, ui->cellWidget(row+1, 5));

	item->setSize(size);

	ui->removeRow(row+1);
}

void VisualizationTextView::moveDownVisualization(Visualization* vis)
{
	VisualizationViewTransformable::moveDownVisualization(vis);

	VisualizationText* v = dynamic_cast<VisualizationText*>(vis);
	assert(v!=NULL);
	IVisualizationTextItem* iv = v->getItem();
	VisualizationTextItem* item = dynamic_cast<VisualizationTextItem*>(iv);
	assert(item!=NULL);
	int row = item->row();

	if (row >= ui->rowCount()-1)
		return;

	// insert at row+2 = after next row
	ui->insertRow(row+2);

	Size2i size = item->getSize();

	// move contents from row to row+2
	ui->setItem(row+2, 0, ui->takeItem(row, 0));
	ui->setCellWidget(row+2, 1, ui->cellWidget(row, 1));
	ui->setItem(row+2, 2, ui->takeItem(row, 2));
	ui->setItem(row+2, 3, ui->takeItem(row, 3));
	ui->setItem(row+2, 4, ui->takeItem(row, 4));
	ui->setCellWidget(row+2, 5, ui->cellWidget(row, 5));

	item->setSize(size);

	ui->removeRow(row);}

const Class& VisualizationTextView::supportedVisualizationClass() const
{
	return VisualizationText::CLASS();
}

const Class& VisualizationTextView::defaultVisualizationClass() const
{
	return DefaultTextVisualization::CLASS();
}

void VisualizationTextView::update(Duration dt)
{
	assert(ui!=NULL);
}

QWidget* VisualizationTextView::createVisualizationPart()
{
	ui = new UI(this);
	setupVisualizations();
	return ui;
}

IVisualizationTextItem* VisualizationTextView::addItem()
{
	assert(ui!=NULL);

	int row = ui->rowCount();
	ui->insertRow(row);

	return new VisualizationTextItem(ui,row);
}

void VisualizationTextView::destroyItem(IVisualizationTextItem* item)
{
	assert(item!=NULL);
	assert(ui!=NULL);
	VisualizationTextItem* it = dynamic_cast<VisualizationTextItem*>(item);
	assert(it!=NULL);
	ui->removeRow(it->row());
	delete item;
}

void VisualizationTextView::setPrecision(uint precision)
{
	if (precision == mPrecision)
		return;

	mPrecision = precision;
	foreach(Visualization* vis, getVisualizations()) {
		VisualizationText* v = dynamic_cast<VisualizationText*>(vis);
		if (v)
			v->setPrecision(precision);
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION( mira::VisualizationTextView, mira::VisualizationViewTransformable);
MIRA_CLASS_SERIALIZATION( mira::DefaultTextVisualization, mira::VisualizationText);

