/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTextTraceView.C
 *    Implementation of Default text trace visualizations.
 *
 * @author Tim Langner, Christof Schroeter
 * @date   2014/06/26
 **/

#include <QDragMoveEvent>
#include <QHeaderView>
#include <QMenu>
#include <QTextStream>

#include <serialization/DefaultInitializer.h>
#include <serialization/Serialization.h>

#include <views/VisualizationView.h>
#include <views/VisualizationControlPage.h>

#include <widgets/MultiInputDialog.h>
#include <widgets/QtUtils.h>

#include <visualization/VisualizationTextTraceBasic.h>
#include <views/private/VisualizationTextTraceView.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class DefaultTextTraceVisualization : public VisualizationTextTraceBasic<void>
{
MIRA_OBJECT(DefaultTextTraceVisualization)

public:

	typedef VisualizationTextTraceBasic<void> Base;

	DefaultTextTraceVisualization() :
		Base("Channel"), mShowROProperties(true), mStringMapAsObject(true)
	{}

	template <typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Base);
		r.property("ShowReadOnlyProperties", mShowROProperties,
		           "Show objects' read-only properties", true);
		r.property("StringMapAsObject", mStringMapAsObject,
		           "Serialize any map<string, T> to json object "
		           "(enables simple access to map items by key)", true);
	}

	virtual void dataChanged(ChannelRead<void> data)
	{
		if (!this->getSite()->isAcceptingUpdates())
			return;

		try
		{
			json::Value v;
			try {
				JSONSerializer js(mShowROProperties,
				                  this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
				                                           : JSONSerializer::STANDARD);
				data.readJSON(v, js);
			}
			catch(XNotImplemented&)
			{
				data.readJSON(v);
			}
			mItem->setStamped(data->timestamp, data->frameID, data->sequenceID,
			                  json::write(v,true,mPrecision).c_str());
		}
		catch(...)
		{
			int typeID = MIRA_FW.getChannelManager().getTypeId(Base::mChannel.getID());
			if (typeID < 0)
				mItem->setStamped(data->timestamp, data->frameID, data->sequenceID,
				                  "Untyped channel, conversion to json format (from metadata) not possible");
			else
				mItem->setStamped(data->timestamp, data->frameID, data->sequenceID,
				                  "Conversion to json format not supported by channel type");
		}
	}

protected:
	bool mShowROProperties;
	bool mStringMapAsObject;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Text view that may contain different visualization objects which
 * show the actual content of attached channels in a trace/history mode
 */
class VisualizationTextTraceView : public VisualizationViewTransformable, public IVisualizationTextTraceSite
{
MIRA_META_OBJECT(VisualizationTextTraceView,
                 ("Name", "Text Trace View")
                 ("Description", "View for traced text visualizations")
                 ("Category"   , "Visualization")
                 ("Visualization", "mira::VisualizationTextTrace"))

public:
	VisualizationTextTraceView();

	virtual ~VisualizationTextTraceView()
	{
		destroyVisualizations();
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		VisualizationViewTransformable::reflect(r);
		r.member("Default Row Heights",
		         mDefaultHeights, "Default values for row heights");
		r.member("Column Widths",
		         getter(&VisualizationTextTraceView_UI::getColumnWidths, ui),
		         setter(&VisualizationTextTraceView::setColumnWidths, this),
		         "Column widths");
		r.property("Auto Scroll",
		           mAutoScroll, "Automatically scroll to any new item", false);
		r.property("Resize All Rows",
		           mResizeAllRows, "Resizing a row affects all rows (with content in that column)", true);
		r.property("MaxHistory", mMaxHistory, "Maximum history of the trace in milliseconds (0 means infinite)", Duration::seconds(0));
		r.property("Precision", mPrecision,
		           setter(&VisualizationTextTraceView::setPrecision, this),
		           "Precision for text output of float numbers.", 3,
		           PropertyHints::minimum(0));
		r.property("Enforce Time Order",
		           mEnforceTimeOrder, "If true, add items to the table at their correct place in time\n"
		                              "(instead of always at end, even if timestamp is older)", false);
	}

public: // implementation of VisualizationView

	virtual void addVisualization(Visualization* vis);

	virtual void moveUpVisualization(Visualization* vis);
	virtual void moveDownVisualization(Visualization* vis);

	virtual const Class& supportedVisualizationClass() const;

	virtual const Class& defaultVisualizationClass() const;

public:

	virtual Duration getMaxHistory() { return mMaxHistory; }

protected:

	virtual void update(Duration dt);

protected:

	virtual QWidget* createVisualizationPart();

	virtual void saveContentToFile() { ui->saveAsCSV(); }

public: // implementation of IVisualizationTextTraceSite

	virtual IVisualizationTextTraceItem* addItem();
	virtual void destroyItem(IVisualizationTextTraceItem* item);

	virtual bool isAcceptingUpdates() { return mAcceptingUpdates; }

	virtual const std::string& getFixedFrame() const {
		return mFixedFrame.getID();
	}

	virtual const std::string& getCameraFrame() const {
		return mCameraFrame.getID();
	}

private:

	void setDefaultHeight(int row, int col);
	void removeDefaultHeight(int col);
	void setColumnWidths(const std::vector<int> & w) {
		mColumnWidths = w;
	}

	virtual void clipHistory(const Time& timestamp);

private:

	VisualizationTextTraceView_UI* ui;

	// one value for each column except column 0 --> size = last column!
	std::vector<int> mDefaultHeights;

	std::vector<int> mColumnWidths;

	bool mAutoScroll;
	bool mResizeAllRows;
	Duration mMaxHistory;

	bool mEnforceTimeOrder;

	bool mAcceptingUpdates;

private:
	void setPrecision(uint precision);
	uint mPrecision;

private:
	friend class VisualizationTextTraceView_UI;
	friend class VisualizationTextTraceItem;

	void setupVisualizations();
};

///////////////////////////////////////////////////////////////////////////////

VisualizationTextTraceItem::VisualizationTextTraceItem(QTableWidget* t, int column,
                                                       VisualizationTextTraceView* v) :
		QObject(t),
		table(t),
		view(v)
{
	columnItem = new QTableWidgetItem("",QTableWidgetItem::Type);
	table->setHorizontalHeaderItem(column, columnItem);
}

void VisualizationTextTraceItem::setName(const std::string& name)
{
	columnItem->setText(name.c_str());
}

void VisualizationTextTraceItem::setStamped(const Time& timestamp, const std::string& frameID,
                                            uint32 sequenceID, const std::string& text)
{
	assert(table!=NULL);
	view->clipHistory(timestamp);
	int row = table->rowCount();

	if (view->mEnforceTimeOrder)
		while ((row > 0) && (table->item(row-1, 0)->data(Qt::UserRole).toULongLong() > timestamp.toUnixNS()))
			--row;

	if (view->mEnforceTimeOrder && (row > 0) &&
            (table->item(row-1, 0)->data(Qt::UserRole).toULongLong() == timestamp.toUnixNS()))
		--row;
	else {
		table->insertRow(row);
		QTableWidgetItem* timestampItem = new QTableWidgetItem();
		timestampItem->setTextAlignment(Qt::AlignLeft | Qt::AlignTop);
		table->setItem(row, 0, timestampItem);
		timestampItem->setText(toString(timestamp.toLocal()).c_str());
		timestampItem->setData(Qt::UserRole, (qulonglong)timestamp.toUnixNS());
	}
	int col = column();
	QTableWidgetItem* textItem = new QTableWidgetItem();
	textItem->setTextAlignment(Qt::AlignLeft | Qt::AlignTop);
	table->setItem(row, col, textItem);
	textItem->setText(text.c_str());
	view->setDefaultHeight(row, col);

	if (view->mAutoScroll)
		table->scrollToItem(textItem);
}

int VisualizationTextTraceItem::column() const
{
	for(int c=0; c<table->columnCount(); ++c)
		if (table->horizontalHeaderItem(c) == columnItem)
		{
			return c;
		}
	return -1;
}

///////////////////////////////////////////////////////////////////////////////

VisualizationTextTraceView_UI::VisualizationTextTraceView_UI(VisualizationTextTraceView* parent) : QTableWidget(parent), vis(parent)
{
	setColumnCount(1);
	setAcceptDrops(true);
	setSelectionMode(QAbstractItemView::NoSelection);
	setHorizontalHeaderLabels(QStringList() << "Time");
	setColumnWidth(1, 200);

	connect((QObject*)verticalHeader(), SIGNAL(sectionResized(int, int, int)),
	        this, SLOT(rowResized(int, int, int)));

	this->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(this,SIGNAL(customContextMenuRequested(const QPoint& )),this,SLOT(popupMenu(const QPoint&)));

	stylesheet = vis->styleSheet();

	vis->startUpdateTimer();
}

VisualizationTextTraceView_UI::~VisualizationTextTraceView_UI()
{
}

void VisualizationTextTraceView_UI::dragEnterEvent(QDragEnterEvent *event)	{
	setAcceptDrops(true);
	vis->dragEnterEventImpl(event);
}

void VisualizationTextTraceView_UI::dropEvent(QDropEvent *event) {
	vis->dropEventImpl(event, this);
}

void VisualizationTextTraceView_UI::dragMoveEvent(QDragMoveEvent* event) {
	event->accept();
}

void VisualizationTextTraceView_UI::rowResized(int row, int oldSize, int newSize)
{
	if (!vis->mResizeAllRows)
		return;

	vis->mResizeAllRows = false; // prevent loopy resize (efficiency)
	for (int c = 1; c < columnCount(); ++c)
	{
		if (item(row, c))
		{
			for (int r = 0; r < rowCount(); ++r)
			{
				if (item(r, c))
					verticalHeader()->resizeSection(r, newSize);
			}
			vis->mDefaultHeights[c-1] = newSize;
		}
	}
	vis->mResizeAllRows = true; // reactivate
}

std::vector<int> VisualizationTextTraceView_UI::getColumnWidths()
{
	std::vector<int> w(columnCount());
	for (int c = 0; c < columnCount(); ++c)
	{
		w[c] = horizontalHeader()->sectionSize(c);
	}
	return w;
}

void VisualizationTextTraceView_UI::popupMenu(const QPoint& pos)
{
	QMenu menu;

	QAction* clearAction = menu.addAction("Clear");
	QAction* csvAction = menu.addAction("Save as CSV");

	if (rowCount() == 0) {
		clearAction->setEnabled(false);
		csvAction->setEnabled(false);
	}

	QAction* freezeAction = menu.addAction("Freeze");
	freezeAction->setCheckable(true);
	freezeAction->setChecked(!vis->mAcceptingUpdates);

	QAction* a = menu.exec(QCursor::pos());
	if (a)
	{
		if (a->text() == "Clear")
		{
			while (rowCount() > 0)
				removeRow(0);
		}

		if (a->text() == "Save as CSV")
		{
			saveAsCSV();
		}

	}

	freeze(freezeAction->isChecked());
}

void VisualizationTextTraceView_UI::freeze(bool freeze)
{
	vis->mAcceptingUpdates = !freeze;
	vis->setStyleSheet(freeze ? stylesheet + " background-color: #DBEBF8;" : stylesheet);
	update();
}

void VisualizationTextTraceView_UI::saveAsCSV()
{
	if (rowCount() == 0)
		return;

	QStringList extList;
	extList << "csv" << "txt";
	static QString saveDir = ".";
	QString fn = QtUtils::getSaveFileName(NULL, trUtf8("Save as CSV"), saveDir,
	                                      trUtf8("Data (*.csv)")+";;"+trUtf8("Text (*.txt)"),
	                                      NULL, QFileDialog::DontUseNativeDialog, extList);
	if (fn.isEmpty())
		return;

	QFile file(fn);
	if (!file.open(QFile::WriteOnly)) {
		MIRA_THROW(XRuntime, std::string("Failed to open file ") + fn.toStdString() + " for writing.");
	}
		
	QTextStream stream(&file);

	QTableWidgetItem* i = horizontalHeaderItem(0);
	QString text = i ? i->text() : "";
	text.replace("\n", " ");
	text.replace("\t", " ");
	stream  << text;		
	for (int c = 1; c < columnCount(); ++c) {
		QTableWidgetItem* i = horizontalHeaderItem(c);
		QString text = i ? i->text() : "";
		text.replace("\n", " ");
		text.replace("\t", " ");
		stream  << "\t" << text;		
	}
	stream << "\n";

	for (int r = 0; r < rowCount(); ++r) {
		QTableWidgetItem* i = item(r,0);
		QString text = i ? i->text() : "";
		text.replace("\n", " ");
		text.replace("\t", " ");
		stream << text;
		for (int c = 1; c < columnCount(); ++c)	{
			QTableWidgetItem* i = item(r,c);
			QString text = i ? i->text() : "";
			text.replace("\n", " ");
			text.replace("\t", " ");
			stream  << "\t" << text;
		}
		stream << "\n";
	}

	// memorize the selected location
	saveDir = QFileInfo(fn).path();
}

///////////////////////////////////////////////////////////////////////////////

VisualizationTextTraceView::VisualizationTextTraceView() :
	ui(NULL), mAutoScroll(false),
	mResizeAllRows(true), mAcceptingUpdates(true), mPrecision(3)
{
	MIRA_INITIALIZE_THIS;
	setUpdateInterval(40);
}


void VisualizationTextTraceView::setupVisualizations()
{
	// initialize all existing visualizations
	foreach(Visualization* vis, getVisualizations())
		vis->init(this);
}

void VisualizationTextTraceView::addVisualization(Visualization* vis)
{
	VisualizationTextTrace* v = dynamic_cast<VisualizationTextTrace*>(vis);
	assert(v!=NULL);
	v->init(this);
	v->setPrecision(mPrecision);
	VisualizationView::addVisualization(vis);
}

void VisualizationTextTraceView::moveUpVisualization(Visualization* vis)
{
	VisualizationViewTransformable::moveUpVisualization(vis);

	VisualizationTextTrace* v = dynamic_cast<VisualizationTextTrace*>(vis);
	assert(v!=NULL);
	IVisualizationTextTraceItem* iv = v->getItem();
	VisualizationTextTraceItem* item = dynamic_cast<VisualizationTextTraceItem*>(iv);
	assert(item!=NULL);
	int col = item->column();

	if (col < 2)
		return;

	// insert at col-1  = before previous column -> col moves to col+1
	ui->insertColumn(col-1);

	ui->setHorizontalHeaderItem(col-1, ui->takeHorizontalHeaderItem(col+1));

	for (int r = 0; r < ui->rowCount(); ++r) {
		QTableWidgetItem* i = ui->takeItem(r, col+1);
		if (i)
			ui->setItem(r, col-1, i);
	}

	ui->horizontalHeader()->resizeSection(col-1, ui->horizontalHeader()->sectionSize(col+1));
	ui->removeColumn(col+1);

	// indices = column-1!
	std::swap(mDefaultHeights[col-1], mDefaultHeights[col-2]);
}
void VisualizationTextTraceView::moveDownVisualization(Visualization* vis)
{
	VisualizationViewTransformable::moveDownVisualization(vis);

	VisualizationTextTrace* v = dynamic_cast<VisualizationTextTrace*>(vis);
	assert(v!=NULL);	
	IVisualizationTextTraceItem* iv = v->getItem();
	VisualizationTextTraceItem* item = dynamic_cast<VisualizationTextTraceItem*>(iv);
	assert(item!=NULL);
	int col = item->column();

	if (col >= ui->columnCount()-1)
		return;

	// insert at col+2  = after next column
	ui->insertColumn(col+2);

	ui->setHorizontalHeaderItem(col+2, ui->takeHorizontalHeaderItem(col));

	for (int r = 0; r < ui->rowCount(); ++r) {
		QTableWidgetItem* i = ui->takeItem(r, col);
		if (i)
			ui->setItem(r, col+2, i);
	}

	ui->horizontalHeader()->resizeSection(col+2, ui->horizontalHeader()->sectionSize(col));
	ui->removeColumn(col);

	// indices = column-1!
	std::swap(mDefaultHeights[col-1], mDefaultHeights[col]);
}

IVisualizationTextTraceItem* VisualizationTextTraceView::addItem()
{
	assert(ui!=NULL);
	ui->setColumnCount(ui->columnCount()+1);
	return new VisualizationTextTraceItem(ui, ui->columnCount()-1, this);
}

void VisualizationTextTraceView::destroyItem(IVisualizationTextTraceItem* item)
{
	assert(ui!=NULL);
	VisualizationTextTraceItem* i = dynamic_cast<VisualizationTextTraceItem*>(item);
	assert(i!=NULL);

	int c = i->column();
	ui->removeColumn(c);
	removeDefaultHeight(c);

	if (ui->columnCount() == 1)
		ui->setRowCount(0);
	else
	{
		// remove resulting empty lines
		for(int r=ui->rowCount()-1; r >= 0; --r)
		{
			bool hasItem = false;
			for (int c = 1; c < ui->columnCount(); ++c)
			{
				if (ui->item(r, c))
				{
					hasItem = true;
					break;
				}
			}
			if (!hasItem)
				ui->removeRow(r);
		}
	}
}

void VisualizationTextTraceView::clipHistory(const Time& timestamp)
{
	if (mMaxHistory == Duration::seconds(0))
		return;

	while (ui->rowCount() > 0)
	{
		if (ui->item(0, 0)->data(Qt::UserRole).toULongLong() < (timestamp - mMaxHistory).toUnixNS())
			ui->removeRow(0);
		else
			break;
	}
}

void VisualizationTextTraceView::setDefaultHeight(int row, int col)
{
	if ((int)mDefaultHeights.size() < col)
		mDefaultHeights.resize(col, ui->verticalHeader()->sectionSize(row));
	else
		ui->verticalHeader()->resizeSection(row, mDefaultHeights[col-1]);
}

void VisualizationTextTraceView::removeDefaultHeight(int col)
{
	if (col > (int)mDefaultHeights.size())
		return;

	for (int i = col-1; i < (int)mDefaultHeights.size()-1; ++i)
		mDefaultHeights[i] = mDefaultHeights[i+1];

	mDefaultHeights.resize(mDefaultHeights.size()-1);
}

const Class& VisualizationTextTraceView::supportedVisualizationClass() const
{
	return VisualizationTextTrace::CLASS();
}

const Class& VisualizationTextTraceView::defaultVisualizationClass() const
{
	return DefaultTextTraceVisualization::CLASS();
}

void VisualizationTextTraceView::update(Duration dt)
{
	assert(ui!=NULL);
}

QWidget* VisualizationTextTraceView::createVisualizationPart()
{
	ui = new VisualizationTextTraceView_UI(this);
	setupVisualizations();
	if (ui->columnCount() == (int)mColumnWidths.size())
	{
		for (int c = 0; c < (int)mColumnWidths.size(); ++c)
			ui->horizontalHeader()->resizeSection(c, mColumnWidths[c]);
	}

	return ui;
}

void VisualizationTextTraceView::setPrecision(uint precision)
{
	mPrecision = precision;
	foreach(Visualization* vis, getVisualizations()) {
		VisualizationTextTrace* v = dynamic_cast<VisualizationTextTrace*>(vis);
		if (v)
			v->setPrecision(precision);
	}
}
///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION( mira::VisualizationTextTraceView, mira::VisualizationViewTransformable);
MIRA_CLASS_SERIALIZATION( mira::DefaultTextTraceVisualization, mira::VisualizationTextTrace);

