/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTextTrace.h
 *    Declaration of VisualizationTextTrace and related classes.
 *
 * @author Tim Langner
 * @date   2014/06/26
 */

#ifndef _MIRA_VISUALIZATIONTEXTTRACE_H_
#define _MIRA_VISUALIZATIONTEXTTRACE_H_

#include <visualization/Visualization.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class IVisualizationTextTraceItem
{
public:
	virtual ~IVisualizationTextTraceItem() {}
public:
	virtual void setName(const std::string& name) = 0;
	virtual void setStamped(const Time& timestamp, const std::string& frameID,
	                        uint32 sequenceID, const std::string& text) = 0;
	template <typename T>
	void setStamped(const Stamped<T>& stamped) {
		setStamped(stamped.timestamp, stamped.frameID, stamped.sequenceID);
	}
};

class IVisualizationTextTraceSite : public IVisualizationSiteTransformable
{
public:

	virtual IVisualizationTextTraceItem* addItem() = 0;
	virtual void destroyItem(IVisualizationTextTraceItem* item) = 0;

	virtual void clipHistory(const Time& timestamp) = 0;

	virtual bool isAcceptingUpdates() { return true; }
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for a text visualization tracing the data history.
 * In contrast to 2D and 3D visualizations, text visualizations provide the
 * "visualization" for their data as text string.
 */
class MIRA_GUI_VISUALIZATION_EXPORT VisualizationTextTrace :
	public Visualization
{
	MIRA_OBJECT(VisualizationTextTrace)
public:

	VisualizationTextTrace();
	virtual ~VisualizationTextTrace() {}

	template <typename Reflector>
	void reflect(Reflector& r) {
		Visualization::reflect(r);

		// don't need to reflect mPrecision as that is stored
		// in the site and forwarded to all visualizations
	}

public:

	/**
	 * Is called by the visualization view when the visualization is attached.
	 * The default implementation stores the pointer to the IVisualizationTextTraceSite
	 * interface that can be retrieved using the getSite() method for later use.
	 * Moreover, it calls the setupScene() method.
	 * You should never need to overwrite this method in derived classes. Use
	 * the setupScene() method instead. However, if you overwrite this method
	 * you SHOULD call this default implementation.
	 *
	 * The passed interface must be of the type IVisualizationTextTraceSite.
	 */
	virtual void init(IVisualizationSite* site);


public:

	/**
	 * Returns a pointer to the IVisualizationTextTraceSite interface that was set
	 * in the init() method.
	 * Or returns NULL, if init() was not yet called.
	 */
	virtual IVisualizationTextTraceSite* getSite();

public:

	/**
	 * Must be implemented in derived classes. Is called within the
	 * visualization's init() method. You can use this method to create
	 * your scene and scene objects.
	 * The passed IVisualizationTextTraceSite interface is the same that is
	 * returned by getSite().
	 */
	virtual void setupScene(IVisualizationTextTraceSite* site) = 0;

public:

	/**
	 * Set precision for displaying floating point numbers.
	 */
	void setPrecision(uint precision);

	/**
	 * Get the visualization item.
	 */
	virtual IVisualizationTextTraceItem* getItem() = 0;

protected:

	uint mPrecision;

private:

	IVisualizationTextTraceSite* mSite;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
