/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization2DView.h
 *    Declaration of the Visualization2DView.
 *
 * @author Tim Langner
 * @date   2011/02/06
 */

#ifndef _MIRA_VISUALIZATION2DVIEW_H_
#define _MIRA_VISUALIZATION2DVIEW_H_

#include <serialization/Serialization.h>
#include <serialization/adapters/Qt/QColor>

#include <views/VisualizationView.h>
#include <visualization/Visualization2D.h>

#include <QGraphicsView>
#include <QMessageBox>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * 2D view of a scene that may contain different visualization objects which
 * show the actual content of the scene.
 */
class Visualization2DView : public VisualizationViewTransformable, public IVisualization2DSite
{
MIRA_META_OBJECT(Visualization2DView,
				("Name", "2D")
				("Description", "2D view for channel data")
				("Category"   , "Visualization")
				("Visualization", "mira::Visualization2D"))

public:
	Visualization2DView();

	virtual ~Visualization2DView()
	{
		// we must make sure that we destroy our tools and visualizations
		// before we destroy ourself to give them the chance to cleanup
		// (during cleanup they may need a fully operational Visualization2DView)
		destroyVisualizations();
		destroyTools();
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		VisualizationViewTransformable::reflect(r);
		r.property("BackgroundColor", mBackgroundColor, setter<QColor>(&Visualization2DView::setBackgroundColor, this), "The color", QColor(Qt::white));
		r.property("Camera", mCameraSettings, "Camera view", Camera());

		r.property("SaveContentMinPictureSize", mSaveContentMinPictureSize,
				   "Minimum width or height of the image saved by rendering the viewport at an appropriate resolution.", 2048);
	}

public: // implementation of VisualizationView

	virtual void addVisualization(Visualization* vis);
	virtual void moveUpVisualization(Visualization* vis);
	virtual void moveDownVisualization(Visualization* vis);

	virtual void addTool(VisualizationTool* tool);

	virtual const Class& supportedVisualizationClass() const;
	virtual const Class& supportedVisualizationToolClass() const;

protected:

	virtual void update(Duration dt);

protected:

	virtual QWidget* createVisualizationPart();
	virtual void resetView();
	virtual void saveContentToFile();

public: // implementation of IVisualization2DSite

	virtual QGraphicsScene* getSceneManager();

	virtual QGraphicsView* getViewManager();

	virtual const std::string& getFixedFrame() const {
		return mFixedFrame.getID();
	}

	virtual const std::string& getCameraFrame() const {
		return mCameraFrame.getID();
	}

	virtual const Camera& getCamera() const;
	virtual void setCamera(const Camera& camera);

	virtual float getToolBaseZValue() const;

private:

	void setBackgroundColor(const QColor& color);
	void updateCamera();

private:

	class UI;
	UI* ui;

private:
	friend class UI;

	void setupVisualizations();
	void setupTools();

private:
	void populateDefaultVisualizations();
	void createDefaultVisualization(const std::string& className, const std::string& displayName);

private:

	QGraphicsScene* mScene;
	QColor mBackgroundColor;
	Camera mCameraSettings;

	uint mSaveContentMinPictureSize;
};

///////////////////////////////////////////////////////////////////////////////

class Visualization2DView::UI : public QGraphicsView
{
	Q_OBJECT
public:

	UI(Visualization2DView* parent);

	virtual ~UI()
	{
	}

	void setCenter(const QPointF& centerPoint);
	QPointF getCenter() const { return mCurrentCenterPoint; }

	void readdVisualizationItems();

protected:

	virtual void dragEnterEvent(QDragEnterEvent *event);

	virtual void dropEvent(QDropEvent *event);

	// we need to overwrite this to prevent the QGraphicsView from disturbing our drag n drop
	virtual void dragMoveEvent(QDragMoveEvent* event);
	virtual void mousePressEvent(QMouseEvent* event);
	virtual void mouseMoveEvent(QMouseEvent* event);
	virtual void mouseReleaseEvent(QMouseEvent* event);
	virtual void wheelEvent (QWheelEvent* event);

protected:
	Visualization2DView* vis;

private:
	QPointF mCurrentCenterPoint;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
