/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkWithGui.h
 *    Provides a framework with Qt support.
 *
 * @author Erik Einhorn
 * @date   2010/10/07
 */

#ifndef _MIRA_FRAMEWORKWITHGUI_H_
#define _MIRA_FRAMEWORKWITHGUI_H_

#ifndef Q_MOC_RUN
#include <boost/scoped_ptr.hpp>
#endif

#include <QApplication>
#include <QMessageBox>
#include <error/LoggingAux.h>
#include <fw/Framework.h>
#include <thread/ThreadMonitor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class FrameworkWithGui : public Framework
{
public:

	/**
	 * Implements our own Application derived from QApplication in order to
	 * overwrite the notify method.
	 * In this method we catch exceptions and show a MessageBox instead of
	 * terminating the whole process (which is the default behavior of
	 * QApplication ...)
	 */
	class Application: public QApplication
	{
	public:

		Application(int& argc, char** argv): QApplication(argc,argv)
		{}

		/**
		 * Overwrites QApplication::notify in order to catch exceptions.
		 */
		bool notify(QObject *rec, QEvent *ev)
		{
			try {
				return QApplication::notify(rec, ev);
			}
			catch (std::exception& ex) {
				MIRA_LOG_EXCEPTION(ERROR, ex) << "Unhandled exception:\n";
				std::string msg = ex.what() +
						std::string("\n\nDo you want to continue or abort");
				if(QMessageBox::critical(NULL, "Unhandled exception", msg.c_str() ,
						QMessageBox::Ignore | QMessageBox::Abort) == QMessageBox::Abort)
					abort();
			}
			catch (...) {
				MIRA_LOG(ERROR) << "Unhandled exception: <unknown exception>";
				std::string msg = "<unknown exception>" +
						std::string("\n\nDo you want to continue or abort");
				if(QMessageBox::critical(NULL, "Unhandled exception", msg.c_str() ,
						QMessageBox::Ignore | QMessageBox::Abort) == QMessageBox::Abort)
					abort();
			}
			return true;
		}
	};

public:

	/// Constructor that creates own Qt application
	FrameworkWithGui(int& argc, char** argv, bool startImmediately = false) :
		Framework(argc, argv, startImmediately),
		mQApp(new Application(argc, argv)),
		mCreatedApp(true)
	{}

	/**
	 * Constructor that takes previously created Qt FrameworkWithGui::Application.
	 * The caller is responsible for destroying the application at the end.
	 */

	FrameworkWithGui(int& argc, char** argv, QApplication* app, bool startImmediately = false) :
		Framework(argc, argv, startImmediately),
		mQApp(app),
		mCreatedApp(false)
	{}

	~FrameworkWithGui()
	{
		if(mCreatedApp)
			delete mQApp;
	}

public:

	/**
	 * Executes the framework and executes the underlying QApplication
	 * until the application is terminated, e.g. by closing it or by
	 * pressing Ctrl+C.
	 */
	virtual int exec() {
		ThreadMonitor::instance().addThisThread("Main Qt thread");
		mInExec = true;
		mQApp->exec();
		mInExec = false;
		MIRA_LOG(DEBUG) << "Application was closed, terminating ...";

		return getTerminationExitCode();
	}

	virtual void requestTermination(int exitcode=0) {
		QCoreApplication::instance ()->quit();
		Framework::requestTermination(exitcode);
	}

private:

	QApplication* mQApp;
	bool mCreatedApp;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
