/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LoggerLoader.C
 *    Implementation of LoggerLoader.h.
 *
 * @author Erik Einhorn
 * @date   2010/11/30
 */

#include <boost/lexical_cast.hpp>

#include <error/LoggingCore.h>
#include <math/Saturate.h>
#include <factory/Factory.h>
#include <xml/XMLDom.h>

#include <loader/Loader.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin used to load configurations for the logger.
 */
class LoggerLoader : public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(LoggerLoader)
public:
	virtual std::vector<std::string> getTags() const {
		return boost::assign::list_of("log");
	}

	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader);
};

void LoggerLoader::parseNode(const XMLDom::const_iterator& node,
                             ConfigurationLoader* loader)
{
	auto itLevel = node.find_attribute("level");
	if (itLevel == node.attribute_end())
		MIRA_THROW(XInvalidConfig, "Error in <log>. "
		           "No level attribute specified. "
		           "Usage <log level=\"1\"/> or <log level=\"DEBUG\"/>.");
	SeverityLevel level;
	try
	{
		int nLevel = boost::lexical_cast<int>(itLevel.value());
		nLevel = saturate<int>(nLevel, CRITICAL, TRACE);
		level = (SeverityLevel)nLevel;
	}
	catch (...) // not a number
	{
		// try as string instead
		level = (SeverityLevel)stringToSeverityLevel(itLevel.value());
	}
	MIRA_LOGGER.setSeverityLevel(level);

}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::LoggerLoader, mira::ConfigurationLoaderPlugin)
