/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteConnection.h
 *    Connection classes representing connections between frameworks.
 *
 * @author Tim Langner
 * @date   2010/11/16
 */

#ifndef _MIRA_REMOTECONNECTION_H_
#define _MIRA_REMOTECONNECTION_H_

#ifndef Q_MOC_RUN
#include <boost/array.hpp>
#endif

#include <utils/UUID.h>
#include <communication/IOService.h>
#include <error/Exceptions.h>

#include <rpc/RPCManager.h>

#include <fw/RemoteAuthority.h>
#include <fw/AuthorityDescription.h>
#include <fw/Channel.h>
#include <fw/FrameworkMessage.h>
#include <fw/ServiceLevel.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// forward declaration
class MIRA_FRAMEWORK_EXPORT Authority;
class MIRA_FRAMEWORK_EXPORT MicroUnit;

/**
 * Informations about a known remote framework
 */
struct KnownFramework
{
	KnownFramework() :
		address(""),
		keep(true),
		lastConnectionTry(Time::unixEpoch()) {}

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.delegate(address);
	}

	void validate() const;
	boost::tuple<std::string, uint16> getHostPort() const;

	/// address in the form of host:port
	std::string address;

	/**
	 * if true the information is stored in list of frameworks
	 * that we try to reconnect to after disconnect
	 */ 
	bool keep;

	/**
	 * The last time we tried to connect to that address. Used for
	 * waiting at least some seconds before trying to connect again.
	 */
	Time lastConnectionTry;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class of connections between frameworks
 */
class RemoteConnection
{
public:

	typedef std::map<std::string, Typename> ChannelTypeMap;
	typedef std::set<std::string> StringSet;
	typedef std::map<std::string, bool> ChannelSendMap;
	typedef std::set<std::string> MetaSet;
	typedef std::list<AuthorityDescription> AuthorityDescriptions;

    ///////////////////////////////////////////////////////////////////////////

	/**
	 * RPCHandler for getting notified when a rpc call on server side is
	 * finished and the response is ready.
	 * Will send the response back to the calling framework using the
	 * connection pointer.
	 */
	class RPCRemoteFinishHandler : public RPCManager::RemoteFinishHandler
	{
	public:
		/// Constructor taking the connection pointer
		RPCRemoteFinishHandler(RemoteConnection* iConnection) :
			mConnection(iConnection)
		{}

		/**
		 * Implementation of RPCManager::RemoteFinishHandler
		 * Will send answer back to calling framework using the internal
		 * connection pointer (if set)
		 */
		virtual void onRPCfinished(const Buffer<uint8>& answer);

		/// Set the connection pointer. Used to reset the connection on disconnect
		void setConnection(RemoteConnection* iConnection);

	protected:

		boost::mutex mConnectionMutex;
		RemoteConnection* mConnection;
	};

    ///////////////////////////////////////////////////////////////////////////

	/**
	 * RPCHandler for sending a rpc call to the server side.
	 * Will send the request to the framework that will process the rpc call
	 * using the connection pointer.
	 */
	class RPCRemoteRequestHandler : public RPCManager::RemoteRequestHandler
	{
	public:
		/// Constructor taking the connection pointer
		RPCRemoteRequestHandler(RemoteConnection* iConnection) :
			mConnection(iConnection)
		{}

		/**
		 * Implementation of RPCManager::RemoteRequestHandler
		 * Will send request to the framework that will process the call
		 * using the internal connection pointer (if set)
		 */
		virtual void onRPCrequested(const Buffer<uint8>& request);

		/// Set the connection pointer. Used to reset the connection on disconnect
		void setConnection(RemoteConnection* iConnection);

	protected:

		boost::mutex mConnectionMutex;
		RemoteConnection* mConnection;
	};

	///////////////////////////////////////////////////////////////////////////

	/// Constructs a remote connection that uses its own io service
	RemoteConnection();

	/// Constructs a remote connection that uses a given io service
	RemoteConnection(boost::asio::io_service& service);

	/// Destructor
	virtual ~RemoteConnection();

	/// Returns the network socket of this connection
	boost::asio::ip::tcp::socket& getSocket()
	{
		return mSocket;
	}

	/// Returns true if this is a local connection between two frameworks on the same machine
	bool isLocal() const;

	/// Starts the connection. Can be implemented in derived classes.
	virtual void start();

	/// Called when writing to the socket failed. Can be implemented in derived classes.
	virtual void onWriteError(boost::system::system_error& e) {}

	/// Close the socket
	void stop();

	virtual void onStop() {}

	/**
	 * Time synchronization between frameworks.
	 *
	 * For local connected frameworks the method synchronizeFrameworks will be
	 * called. For frameworks on remote hosts sendPTP will be called.
	 */
	void syncTime();

	/**
	 * Sends a PTP command used for time synchronization between frameworks
	 */
	void sendPTP();

	/**
	 * Sends a ping command. This method ensures, that the ping is not send
	 * more than the configured interval in the RemoteModule.
	 */
	void ping();

	/**
	 * Check if the connection incoming ping's are still alive.
	 */
	bool hasPingTimeout() const;

	/**
	 * Notifies the connected framework that we have at least one publisher for
	 * each of the channels in the channels map.
	 */
	void publishChannels(const ChannelTypeMap& channels);

	/**
	 * Notifies the connected framework that we no longer have a publisher for
	 * the given channel.
	 */
	void unpublishChannel(const std::string& channel);

	/**
	 * Notifies the connected framework that the authorities in the authorities
	 * map exist in our framework
	 */
	void publishAuthorities(const AuthorityDescriptions& authorities);

	/**
	 * Notifies the connected framework that the authorities in the authorities
	 * map do not longer exist in our framework
	 */
	void unpublishAuthorities(const AuthorityDescriptions& authorities);

	/**
	 * Send a request to the connected framework to transfer ownership of a unit
	 * to this framework.
	 */
	void migrateUnit(const std::string& id);

	/**
	 * Check if a authority with given full id exists in the connected
	 * framework
	 */
	bool hasAuthority(const std::string& id) const;

	/**
	 * Notifies the connected framework that the services in the services
	 * set exist in our framework
	 */
	void publishServices(const StringSet& services);

	/**
	 * Notifies the connected framework that the services in the services
	 * set do not longer exist in our framework
	 */
	void unpublishServices(const StringSet& services);

	/**
	 * Send data in the buffers to the connected framework
	 * @param[in] buffers The data to send
	 */
	template <typename BufferSequence>
	void write(const BufferSequence& buffers)
	{
		boost::mutex::scoped_lock lock(mWriteMutex);
		if (mStopped)
			return;
		try
		{
			std::size_t bytes = boost::asio::write(mSocket, buffers, boost::asio::transfer_all());
			updateOutgoingStats(bytes);
		}
		catch (boost::system::system_error& e)
		{
			// let the derived class decide what to do with the error
			onWriteError(e);
		}
	}

	/**
	 * Writes a message to the other framework.
	 * This method takes the message type and a variable number of optional
	 * parameters that are sent within the message.
	 */
	void writeMessage(FrameworkMessageType msg) {
		FrameworkMessageHeader header(0, msg);
		boost::array<boost::asio::const_buffer, 1> buffers =
		{{
			header.asBuffer(),
		}};
		write(buffers);
	}

	///@cond INTERNAL

	/**
	 * Generates the writeMessage(FrameworkMessageType msg, param1 ... paramN)
	 * methods for multiple parameter counts using boost preprocessor.
	 */
	#define GEN_WRITEMESSAGE_WRITE(z,n,_) os << p##n;
	#define GEN_WRITEMESSAGE_PARAM_DECL(z,n,_) , const P##n & p##n
	#define GEN_WRITEMESSAGE(z,n,_)                                            \
	template<BOOST_PP_ENUM_PARAMS_Z(z,BOOST_PP_INC(n),typename P)>             \
	void writeMessage(FrameworkMessageType msg                                 \
	  BOOST_PP_REPEAT_ ## z(BOOST_PP_INC(n),GEN_WRITEMESSAGE_PARAM_DECL,nil)) {\
		BinaryBufferOstream::buffer_type buffer;                               \
		BinaryBufferOstream os(&buffer);                                       \
		BOOST_PP_REPEAT_ ## z(BOOST_PP_INC(n),GEN_WRITEMESSAGE_WRITE,nil)      \
		FrameworkMessageHeader header(buffer.size(), msg);                     \
		boost::array<boost::asio::const_buffer, 2> buffers =                   \
		{{                                                                     \
			header.asBuffer(),                                                 \
			boost::asio::buffer(buffer.data(), buffer.size())                  \
		}};                                                                    \
		write(buffers);                                                        \
	}
	BOOST_PP_REPEAT(BOOST_PP_INC(8),GEN_WRITEMESSAGE, nil);
	#undef GEN_WRITEMESSAGE
	#undef GEN_WRITEMESSAGE_WRITE
	#undef GEN_WRITEMESSAGE_PARAM_DECL

	///@endcond

	/**
	 * Channel callback method that gets registered on each channel the
	 * connected framework subscribes.
	 * When data in the channel changes it is sent to the connected framework.
	 */
	void valueChanged(ChannelRead<void> value, ServiceLevel& serviceLevel);

	/**
	 * Parses an incoming message (stored in mMessage) and calls the
	 * respective receivedXXX method.
	 */
	void parseMessage();

	KnownFramework address;       ///< The address of the connected framework
	UUID remoteID;                ///< The UUID of the connected framework
	std::string frameworkID;      ///< The ID/Name of the connected framework
	uint32 remoteVersion;         ///< The protocol version of the connected framework
	Authority* authority;         ///< Our authority used for subscribing to data
	ChannelSendMap subscriptions; ///< List of channels the connected framework is subscribed to
	MetaSet sentMetaInformation;  ///< Set of type meta information already sent
	StringSet publishedServices;  ///< List of services of the connected framework

	bool synchronized;            ///< Is the connection fully established (e.g. PTP synchronized)
	Duration masterSlaveOffset;   ///< The time offset between us and the connected framework

private:

	void init(); // called by constructors to initialize members

protected:

	void receivedPTPFollowUP(uint64 timestamp);
	void receivedPTPDelayResponse(uint64 timestamp);
	void receivedPTPDelayRequest(uint64 timestamp);
	void receivedPTPFinish();
	void receivedSubscribeChannelRequest();
	void receivedUnsubscribeChannelRequest(const std::string& channelID);
	void receivedPublishChannelMsg();
	void receivedUnpublishChannelMsg();
	void receivedPublishAuthorityMsg();
	void receivedUnpublishAuthorityMsg();
	void receivedPublishServiceMsg();
	void receivedUnpublishServiceMsg();
	void receivedWriteChannelMsg();
	void receivedRequestMigrationMsg();
	void receivedMigrationMsg();
	void receivedMigrationSinkSuccessMsg();
	void receivedMigrationSinkFailureMsg();
	void receivedMigrationFinishedMsg();
	void receivedTypeMetaMsg();
	void receivedChannelMetaMsg();
	void receivedPingMsg();

	void sendData(ChannelRead<void> value, ServiceLevel& serviceLevel);
	void synchronizeFrameworks();
	void updateOutgoingStats(std::size_t size);

	/**
	 * Returns true, if the message (header) is valid, i.e. if the specified
	 * message length does not exceed a max size and the message type is
	 * known. This is used to prevent Denial of Service attacks.
	 */
	bool checkMessageHeader() const;

	void sendConnectDenied(const std::string& msg);

	void processPingThread();
	void checkPingTimeoutThread();

	IOService mService;
	boost::asio::ip::tcp::socket mSocket;

	boost::mutex mWriteMutex;
	boost::mutex mStopMutex;

	boost::thread mProcessPingThread;
	boost::thread mCheckPingTimeoutThread;

	Time mHeaderReceived;
	FrameworkMessageHeader mHeader;
	Buffer<uint8> mMessage;

	enum AuthState {
		AUTHSTATE_NONE,
		AUTHSTATE_CONNECTING,
		AUTHSTATE_AUTHENTICATING,
		AUTHSTATE_ACCEPTED,
		AUTHSTATE_DENIED,
	};
	AuthState mAuthState;
	std::string mAuthSignMsg; // random message that is used for strong authentication

	Time mPTPSyncSlave;
	Time mPTPSyncMaster;
	Time mPTPDelaySlave;
	Time mPTPDelayMaster;
	Time mLastPTP;

	Time mPingLastSend;
	Time mPingLastReceived;

	boost::shared_ptr<MicroUnit> mMigrationUnit;
	std::string mMigrationNS;
	std::string mMigrationID;
	bool mStopped;

	boost::shared_ptr<RPCRemoteFinishHandler> mRPCFinishHandler;
	boost::shared_ptr<RPCRemoteRequestHandler> mRPCRequestHandler;
	std::map<std::string, boost::shared_ptr<RemoteAuthority>> mRemoteAuthorities;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Connection class for incoming connections.
 */
class RemoteIncomingConnection : public RemoteConnection
{
public:

	RemoteIncomingConnection(boost::asio::io_service& service);

	/// Implementation of RemoteConnection
	virtual void start();

	/// Implementation of RemoteConnection
	virtual void onStop();

	/// Register a callback function that is called when this connection closes
	void setDisconnectCallback(boost::function<void (RemoteIncomingConnection*)> fn)
	{
		mDisconnectedCallback = fn;
	}

	/// Returns the endpoint of the remote peer
	boost::asio::ip::tcp::endpoint& getEndpoint() {
		return mPeerEndpoint;
	}

protected:

	void handleReadHeader(const boost::system::error_code& error);

	void handleReadMessage(const boost::system::error_code& error);

	boost::function<void (RemoteIncomingConnection*)> mDisconnectedCallback;

protected:

	/// endpoint of the remote peer
	boost::asio::ip::tcp::endpoint mPeerEndpoint;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Connection class for outgoing connections.
 */
class RemoteOutgoingConnection : public RemoteConnection
{
public:

	RemoteOutgoingConnection(const KnownFramework& address);

	/// Implementation of RemoteConnection
	virtual void start();

	/// Implementation of RemoteConnection
	virtual void onStop();

	/// Implementation of RemoteConnection
	virtual void onWriteError(boost::system::system_error& e);

protected:

	void handleConnect(const boost::system::error_code& error,
	                   boost::asio::ip::tcp::resolver::iterator iterator);

	void handleReadHeader(const boost::system::error_code& error);

	void handleReadMessage(const boost::system::error_code& error);

	std::string mHostName;
	uint16 mPort;
	bool mStopScheduled;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
