
#include <fw/MicroUnit.h>

#include <deque>

using namespace mira;

namespace tutorials {

///////////////////////////////////////////////////////////////////////////////

/**
 * Computes the moving average within a window for a given float channel.
 */
class MeanCalculator : public MicroUnit
{
MIRA_OBJECT(MeanCalculator)

public:

	MeanCalculator();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MicroUnit::reflect(r);
		r.property("WindowSize", mWindowSize,
		           setter<int>(&MeanCalculator::setWindowSize, this),
		           "The window size for computing the average", 10);
		r.roproperty("QueueSize",
		           getter<int>(&MeanCalculator::getQueueSize, this),
		           "The current size of the window for computing the average");
	}

protected:

	virtual void initialize();

private:
	void onNewData(ChannelRead<float> data);
	void setWindowSize(int newSize);
	int getQueueSize() const;

private:
	std::deque<float> mQueue;
	Channel<float> mMeanChannel;

	int mWindowSize;
};

MeanCalculator::MeanCalculator()
{
}

void MeanCalculator::initialize()
{
	subscribe<float>("FloatChannel", &MeanCalculator::onNewData);
	mMeanChannel = publish<float>("MeanChannel");
}

void MeanCalculator::onNewData(ChannelRead<float> data)
{
	mQueue.push_back(data->value());
	if (mQueue.size() > mWindowSize)
		mQueue.pop_front();
	float sum = 0.0;
	foreach(float f, mQueue)
		sum += f;
	float mean = sum / mQueue.size();

	std::cout << "MeanCalculator: " << mean << std::endl;

	ChannelWrite<float> w = mMeanChannel.write();
	w->value() = mean;
}

void MeanCalculator::setWindowSize(int newSize)
{
	mWindowSize = newSize;
	while (mQueue.size() > mWindowSize)
		mQueue.pop_front();
}

int MeanCalculator::getQueueSize() const
{
	return mQueue.size();
}

}
MIRA_CLASS_SERIALIZATION(tutorials::MeanCalculator, mira::MicroUnit );
