/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UniqueIdentifierTest.C
 *    Test cases for UniqueIdentifier.h.
 *
 * @author Christof Schröter
 * @date 2018/08/29
 *
 */

#include <iostream>

#include <boost/test/unit_test.hpp>

#include <utils/UniqueIdentifier.h>

// construct strings from arguments to avoid warnings about undefined literal comparison
#define CHECK_UNIQUE_STRING(A,B)       \
BOOST_CHECK(std::string(A) != std::string(B));

#define CHECK_UNIQUE_IDENTIFIER(A,B)   \
BOOST_CHECK(std::string(BOOST_PP_STRINGIZE(A)) != std::string(BOOST_PP_STRINGIZE(B)));

#define CHECK_EQUAL_STRING(A,B)       \
BOOST_CHECK(std::string(A) == std::string(B));

#define CHECK_EQUAL_IDENTIFIER(A,B)   \
BOOST_CHECK(std::string(BOOST_PP_STRINGIZE(A)) == std::string(BOOST_PP_STRINGIZE(B)));

#define OUT2(expect, A, B)                            \
std::cout << "a/b are " << expect << ":" << std::endl \
          << "A: '" << A << "'" << std::endl << "B: '" << B << "'" << std::endl << std::endl;

#define CHECK_UNIQUE_IDENTIFIER_INTERNAL \
CHECK_UNIQUE_IDENTIFIER(MIRA_UNIQUE_IDENTIFIER, MIRA_UNIQUE_IDENTIFIER);

#define CHECK_EQUAL_IDENTIFIER_INTERNAL \
CHECK_EQUAL_IDENTIFIER(MIRA_FILE_LINE_IDENTIFIER, \
                                                   \
                                                   \
                        MIRA_FILE_LINE_IDENTIFIER);

void check_unique(const std::string& a, const std::string& b)
{
	CHECK_UNIQUE_STRING(a,b);
}

void check_equal(const std::string& a, const std::string& b)
{
	CHECK_EQUAL_STRING(a,b);
}

void out2(const std::string& expect, const std::string& a, const std::string& b)
{
	OUT2(expect,a,b);
}

#ifdef MIRA_UNIQUE_IDENTIFIER_ENABLED

BOOST_AUTO_TEST_CASE( UniqueIdentifierTest )
{
	std::string s1 = MIRA_UNIQUE_STRING;
	std::string s2 = MIRA_UNIQUE_STRING;
	std::string s3 = MIRA_UNIQUE_STRING, s4 = MIRA_UNIQUE_STRING;

	std::cout << "s1 - s4 are unique:" << std::endl;
	std::cout << "s1: '" << s1 << "'" << std::endl
	          << "s2: '" << s2 << "'" << std::endl
	          << "s3: '" << s3 << "'" << std::endl
	          << "s4: '" << s4 << "'" << std::endl << std::endl;

	BOOST_CHECK(s1 != s2);
	BOOST_CHECK(s1 != s3);
	BOOST_CHECK(s3 != s4);

	CHECK_UNIQUE_IDENTIFIER(MIRA_UNIQUE_IDENTIFIER, MIRA_UNIQUE_IDENTIFIER);
	CHECK_UNIQUE_STRING(MIRA_UNIQUE_STRING, MIRA_UNIQUE_STRING);

	CHECK_UNIQUE_IDENTIFIER_INTERNAL;

	// should generate 2 different valid variable names
	int MIRA_UNIQUE_IDENTIFIER, MIRA_UNIQUE_IDENTIFIER;

	// more unique names
	class MIRA_UNIQUE_IDENTIFIER
	{
	};

	class MIRA_UNIQUE_IDENTIFIER
	{
	};
}

BOOST_AUTO_TEST_CASE( FileLineIdentifierTest )
{
	std::string s1 = MIRA_FILE_LINE_STRING;
	std::string s2 = MIRA_FILE_LINE_STRING;
	std::string s3 = MIRA_FILE_LINE_STRING, s4 = MIRA_FILE_LINE_STRING;

	std::cout << "s1/s2 are unique, s3/s4 are identical:" << std::endl;
	std::cout << "s1: '" << s1 << "'" << std::endl
	          << "s2: '" << s2 << "'" << std::endl
	          << "s3: '" << s3 << "'" << std::endl
	          << "s4: '" << s4 << "'" << std::endl << std::endl;

	BOOST_CHECK(s1 != s2);
	BOOST_CHECK(s1 != s3);
	BOOST_CHECK(s3 == s4);

	CHECK_EQUAL_IDENTIFIER(MIRA_FILE_LINE_IDENTIFIER, MIRA_FILE_LINE_IDENTIFIER);
	CHECK_EQUAL_STRING(MIRA_FILE_LINE_STRING, MIRA_FILE_LINE_STRING);

	// macro arguments are on the same line even when the macro call
	// is written across multiple lines!
	CHECK_EQUAL_IDENTIFIER(MIRA_FILE_LINE_IDENTIFIER,
	                       MIRA_FILE_LINE_IDENTIFIER);
	CHECK_EQUAL_STRING(MIRA_FILE_LINE_STRING,
	                   MIRA_FILE_LINE_STRING);

	OUT2("identical", MIRA_FILE_LINE_STRING, MIRA_FILE_LINE_STRING);
	OUT2("identical", MIRA_FILE_LINE_STRING,
	                  MIRA_FILE_LINE_STRING);

	// in contrast, function arguments can actually be on different lines
	check_equal(MIRA_FILE_LINE_STRING, MIRA_FILE_LINE_STRING);
	check_unique(MIRA_FILE_LINE_STRING,
	             MIRA_FILE_LINE_STRING);

	out2("unique", MIRA_FILE_LINE_STRING,
	               MIRA_FILE_LINE_STRING);

	CHECK_EQUAL_IDENTIFIER_INTERNAL;

	// should clash, compilation error
	//int MIRA_FILE_LINE_IDENTIFIER, MIRA_FILE_LINE_IDENTIFIER;

	// should generate 2 different valid variable names
	int MIRA_FILE_LINE_IDENTIFIER,
	    MIRA_FILE_LINE_IDENTIFIER;

	// more unique names
	class MIRA_FILE_LINE_IDENTIFIER
	{
	};

	class MIRA_FILE_LINE_IDENTIFIER
	{
	};
}

#endif

