/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryJSONConverterTest.C
 *
 * @author Erik Einhorn
 */

#include <iostream>
#include <boost/test/unit_test.hpp>

#include <serialization/BinarySerializer.h>
#include <serialization/BinaryJSONConverter.h>
#include <serialization/MetaSerializer.h>
#include <serialization/Print.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/map>

#include <transform/Pose.h>
#include <utils/Stamped.h>

using namespace mira;

BOOST_AUTO_TEST_CASE( BinaryJSONTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Stamped<std::list<std::vector<Pose2*>*>> v;
	auto typemeta = ms.addMeta(v);

	v.timestamp = Time::now();
	v.frameID = "myframe";

	v.push_back(new std::vector<Pose2*>);
	v.push_back(NULL);

	v.front()->push_back(new Pose2(1.0f,2.0f,3.0f));
	v.front()->push_back(NULL);
	v.front()->push_back(new Pose2(6.0f,2.0f,7.0f));
	//v.front()->push_back(v.front()->back()); // pointer tracking not implemented yet

	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

	BOOST_CHECK(val.type() == json_spirit::obj_type);
	BOOST_CHECK(val.get_obj()["FrameID"].get_str() == "myframe");
	json::Array& a1 = val.get_obj()["Value"].get_array();
	BOOST_CHECK_EQUAL(a1.size(), 2);
	json::Array& a2 = a1[0].get_array();
	BOOST_CHECK_EQUAL(a2.size(), 3);
	json::Object& o1 = a2[0].get_obj();
	BOOST_CHECK_CLOSE(o1["X"].get_real(), 1.0, 0.00001);
	BOOST_CHECK_CLOSE(o1["Y"].get_real(), 2.0, 0.00001);
	BOOST_CHECK_CLOSE(o1["Phi"].get_real(), rad2deg(3.0), 0.00001);
	BOOST_CHECK(a2[1].type() == json_spirit::null_type);
	json::Object& o2 = a2[2].get_obj();
	BOOST_CHECK_CLOSE(o2["X"].get_real(), 6.0, 0.00001);
	BOOST_CHECK_CLOSE(o2["Y"].get_real(), 2.0, 0.00001);
	BOOST_CHECK_CLOSE(o2["Phi"].get_real(), rad2deg(7.0), 0.00001);
	BOOST_CHECK(a1[1].type() == json_spirit::null_type);
}

BOOST_AUTO_TEST_CASE( JSONBinaryTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Stamped<std::list<std::vector<Pose2*>*>> v;
	auto typemeta = ms.addMeta(v);

	v.timestamp = Time::now();
	v.frameID = "myframe";

	v.push_back(new std::vector<Pose2*>);
	v.push_back(NULL);

	v.front()->push_back(new Pose2(1.0f,2.0f,3.0f));
	v.front()->push_back(NULL);
	v.front()->push_back(new Pose2(6.0f,2.0f,7.0f));
	//v.front()->push_back(v.front()->back()); // pointer tracking not implemented yet

	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

	Buffer<uint8> backBuf;
	BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

	BinaryBufferDeserializer bd(&backBuf);
	Stamped<std::list<std::vector<Pose2*>*>> backV;
	bd.deserialize(backV);

	BOOST_CHECK(v.timestamp == backV.timestamp);
	BOOST_CHECK(v.frameID == backV.frameID);
	BOOST_CHECK(v.size() == 2);
	BOOST_CHECK(v.front()->size() == 3);
	BOOST_CHECK(v.front()->at(0)->isApprox(*backV.front()->at(0)));
	BOOST_CHECK(v.front()->at(1) == NULL);
	BOOST_CHECK(v.front()->at(2)->isApprox(*backV.front()->at(2)));
	BOOST_CHECK(v.back() == NULL);
}

BOOST_AUTO_TEST_CASE( BinaryJSONDeserializeTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	std::map<int, int> v;
	auto typemeta = ms.addMeta(v);

	v[0] = 1;
	v[1] = 2;
	v[2] = 3;

	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

	JSONValue val;
	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);

	JSONDeserializer jd(val);

	std::map<int, int> backV;
	jd.deserialize(backV);

	std::cout << "in: " << print(v) << std::endl;
	std::cout << "out: " << print(backV) << std::endl;

	BOOST_CHECK(v == backV);
}

BOOST_AUTO_TEST_CASE( JSONBinarySpecializedReflectTest )
{
	// Test datatype that does have a specialized reflect function
	// for the binary serializer
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Stamped<RigidTransformCov3f> v;
	v.value().x() = 1;
	v.value().y() = 2;
	v.value().z() = 3;
	auto typemeta = ms.addMeta(v);

	// print out database for manual debugging purposes
//	foreach(auto p, db)
//	{
//		std::cout << p.first <<  p.second->toString() << std::endl;
//	}

	v.timestamp = Time::now();
	v.frameID = "myframe";

	JSONValue val;
	Buffer<uint8> buf;
	BinaryBufferSerializer bs(&buf);
	bs.serialize(v);

	BinaryJSONConverter::binaryToJSON(buf, true, *typemeta, db, val);
	Buffer<uint8> backBuf;
	BinaryJSONConverter::JSONToBinary(val, *typemeta, db, backBuf, true);

	BinaryBufferDeserializer bd(&backBuf);
	Stamped<RigidTransformCov3f> backV;
	bd.deserialize(backV);

	BOOST_CHECK(v.timestamp == backV.timestamp);
	BOOST_CHECK(v.frameID == backV.frameID);
	BOOST_CHECK(v.value().x() == backV.value().x());
}


