/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ManifestLoadTest.h
 *    $Try to load manifest files and play with vacant classes.$.
 *
 * @author Ronny Stricker
 * @date   2010/10/25
 */

#include <boost/test/unit_test.hpp>

#include <factory/Factory.h>
#include <factory/ManifestAgent.h>

using namespace std;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/*
 * @brief Check meta info parsing with user defined matching function.
 */
bool checkfunc( std::map<std::string, std::string> const& pMetaMap )
{
	return ( ( pMetaMap.find( "Meta" ) != pMetaMap.end() )
			&& ( pMetaMap.find( "Meta" )->second == "Test" )
			&& ( pMetaMap.find( "Meta2" ) != pMetaMap.end() )
			&& ( pMetaMap.find( "Meta2" )->second == "Test2" ) );
}

BOOST_AUTO_TEST_CASE( DLOpen )
{
	// load the manifest files -> objects are spread over 2 different
	// libraries, which are not loaded up to now
	ManifestAgent tManifestAgent;
	tManifestAgent.loadFile( "testTestLib.mani" );
	tManifestAgent.loadFile( "testTestDepLib.mani" );
	tManifestAgent.finalize();

	// get two ClassProxy objects from the two different libraries
	ClassProxy tLibObject = ClassFactory::getClassByIdentifier("mira::LibObject");
	ClassProxy tDepLibObject = ClassFactory::getClassByIdentifier("mira::DepLibObject");

	// try to get list of classes matching the given meta info criterion
	vector<ClassProxy> tClasses =
			ClassFactory::getClassByMeta( "Meta", "Test" );
	BOOST_CHECK_EQUAL( tClasses.size(), 2 );
	// assure that the classes are not abstract
	foreach( ClassProxy& tClass, tClasses ) {
		BOOST_CHECK_EQUAL( tClass.isAbstract(), false );
	}

	BOOST_CHECK_EQUAL( tLibObject.isLibraryLoaded(), false );
	BOOST_CHECK_EQUAL( tDepLibObject.isLibraryLoaded(), false );

	// try to get list of classes matching the user defined meta info criterion
	tClasses =
			ClassFactory::getClassByMeta( boost::bind(checkfunc,_1) );
	BOOST_CHECK_EQUAL( tClasses.size(), 1 );
	BOOST_CHECK_EQUAL( tClasses[0].getIdentifier(), "mira::DepLibObject" );

	BOOST_CHECK_EQUAL( ClassFactory::getClassByIdentifier("mira::LibObject2")
		.isBaseOf( "mira::DepLibObject" ), true );
	BOOST_CHECK_EQUAL( ClassFactory::getClassByIdentifier("mira::LibObject2")
		.isBaseOf( "mira::DepLibObject" ), true );

	// try to create new instance of LibObject
	// therefore the first library have to be loaded
	Object* tObject = ClassFactory::newInstance<Object>( "mira::LibObject" );

	BOOST_CHECK( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::LibObject" );

	BOOST_CHECK_EQUAL( tLibObject.isLibraryLoaded(), true );
	BOOST_CHECK_EQUAL( tDepLibObject.isLibraryLoaded(), false );

	// check parent - child relations
	BOOST_CHECK_EQUAL( tObject->getClass().isBaseOf( "mira::LibObject2" ), true );
	BOOST_CHECK_EQUAL( tObject->getClass().isBaseOf( "mira::DepLibObject" ), true );

	BOOST_CHECK_EQUAL( ClassFactory::getClassByIdentifier("mira::LibObject2")
		.isBaseOf( "mira::DepLibObject" ), true );
	BOOST_CHECK_EQUAL( ClassFactory::getClassByIdentifier("mira::LibObject2")
		.isBaseOf( "mira::DepLibObject" ), true );

	// get list of derived classes
	map< string, ClassProxy > tChildren
		= tObject->getClass().getDerivedClasses();
	BOOST_CHECK_EQUAL( tChildren.size(), 2 );

	// try to create LibObject2 -> still in first library
	Object* tObject2 = tObject->getClass().newInstance( "mira::LibObject2" );
	BOOST_CHECK( tObject2 != NULL );
	BOOST_CHECK_EQUAL( tObject2->getClass().getIdentifier(), "mira::LibObject2" );

	delete tObject2;

	// try to create DepLibObject -> involves loading of second library
	tObject2 = tObject->getClass().newInstance( "mira::DepLibObject" );
	BOOST_CHECK( tObject2 != NULL );
	BOOST_CHECK_EQUAL( tObject2->getClass().getIdentifier(), "mira::DepLibObject" );

	delete tObject2;
	
	BOOST_CHECK_EQUAL( tLibObject.isLibraryLoaded(), true );
	BOOST_CHECK_EQUAL( tDepLibObject.isLibraryLoaded(), true );

	// try to create DepLibObject a second time
	// pointer to VacantClass should have changed to pointer to TClass in child
	// list of tObject
	tObject2 = tObject->getClass().newInstance( "mira::DepLibObject" );
	BOOST_CHECK( tObject2 != NULL );
	BOOST_CHECK_EQUAL( tObject2->getClass().getIdentifier(), "mira::DepLibObject" );

	delete tObject2;
	delete tObject;

	// try to create LibObject with paremeter constructor
	tObject = ClassFactory::newInstance<Object>( "mira::LibObject", 1, 5 );
	BOOST_CHECK( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::LibObject" );
	delete tObject;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
