/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSASignature.C
 *    Implementation of RSASignature.h
 *    on a RSA encryption.
 *
 * @author Christian Martin
 * @date   2011/01/10
 */

#include <security/RSASignature.h>

#include <openssl/md5.h>
#include <openssl/sha.h>
#include <openssl/ripemd.h>

#include <openssl/err.h>
#include <openssl/objects.h>
#include <openssl/rsa.h>

#include <boost/format.hpp>

#include <utils/ToString.h>

#include "OpenSSLHelper.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

RSASignature::RSASignature() :
	mSize(0),
	mData(NULL)
{
	OpenSSLCleanup::instance();
}

RSASignature::RSASignature(const RSASignature& signature) :
	mSize(signature.mSize),
	mData(NULL)
{
	OpenSSLCleanup::instance();

	mData = new uint8[mSize];
	memcpy(mData, signature.mData, mSize);
}

RSASignature::~RSASignature()
{
	delete [] mData;
	mData = NULL;
	mSize = 0;
}

///////////////////////////////////////////////////////////////////////////////

RSASignature& RSASignature::operator=(const RSASignature& signature)
{
	delete [] mData;
	mSize = signature.mSize;
	mData = new uint8[mSize];
	memcpy(mData, signature.mData, mSize);
	return(*this);
}

///////////////////////////////////////////////////////////////////////////////

RSASignature RSASignature::signMessage(const RSAKey& iPrivateKey,
                                       DigestType iDigestType,
                                       const char* iMsg,
                                       size_t iMsgLen)
{
	if (!iPrivateKey.isPrivateKey())
		MIRA_THROW(XInvalidParameter,
		          "Need a private RSA key to sign a message.");

	if ((iMsg == NULL) || (iMsgLen == 0))
		MIRA_THROW(XInvalidParameter, "Can't sign an empty message.");

	///////////////////////////////////////////////////////////////////////////
	// build the digest of the message

	int tDigestType = 0;
	size_t tDigestLen = 0;
	unsigned char* tDigest = NULL;

	if (iDigestType == DIGEST_MD5) {
		tDigestType = NID_md5;
		tDigestLen = MD5_DIGEST_LENGTH;

		MD5_CTX tCtx;
		MD5_Init(&tCtx);
		MD5_Update(&tCtx, iMsg, iMsgLen);
		tDigest = new unsigned char[tDigestLen];
		MD5_Final(tDigest, &tCtx);
	} else
	if (iDigestType == DIGEST_SHA1) {
		tDigestType = NID_sha1;
		tDigestLen = SHA_DIGEST_LENGTH;

		SHA_CTX tCtx;
		SHA1_Init(&tCtx);
		SHA1_Update(&tCtx, iMsg, iMsgLen);
		tDigest = new unsigned char[tDigestLen];
		SHA1_Final(tDigest, &tCtx);
	} else
	if (iDigestType == DIGEST_RIPEMD160) {
		tDigestType = NID_ripemd160;
		tDigestLen = RIPEMD160_DIGEST_LENGTH;

		RIPEMD160_CTX tCtx;
		RIPEMD160_Init(&tCtx);
		RIPEMD160_Update(&tCtx, iMsg, iMsgLen);
		tDigest = new unsigned char[tDigestLen];
		RIPEMD160_Final(tDigest, &tCtx);
	}

	if (tDigest == NULL)
		MIRA_THROW(XInvalidParameter,
		           "Unable to build message digest. Invalid digest type?");

	///////////////////////////////////////////////////////////////////////////
	// now sign the message

	int tKeySize = RSA_size(iPrivateKey.getOpenSSLKey()->key);
	if (tKeySize < 1) {
		delete [] tDigest;
		MIRA_THROW(XInvalidParameter, "Invalid private key.");
	}

	unsigned int tSigLen = 0;
	uint8* tSignature = new uint8[tKeySize];
	if (RSA_sign(tDigestType, tDigest, tDigestLen,
	             tSignature, &tSigLen, iPrivateKey.getOpenSSLKey()->key) != 1)
	{
		std::string tFullErrMsg;
		unsigned long tErrCode = ERR_get_error();
		do {
			std::string tErrMsg =
				OpenSSLErrorString::instance().err2str(tErrCode);
			if (tFullErrMsg.size() > 0)
				tFullErrMsg += ", ";
			tFullErrMsg += tErrMsg;
		} while ((tErrCode = ERR_get_error()) != 0);

		delete [] tDigest;
		delete [] tSignature;
		MIRA_THROW(XInvalidParameter, "RSA_sign failed: " << tFullErrMsg);
	}

	delete [] tDigest;

	if (tKeySize != (int)tSigLen) {
		delete [] tSignature;
		MIRA_THROW(XInvalidParameter, "Unexpected RSA signature length: "
		           "Size of the signature must be equal to the size of the key "
		           << "(KeySize=" << tKeySize
		           << ", SignatureSize=" << tSigLen << ").");
	}

	///////////////////////////////////////////////////////////////////////////

	RSASignature tResult;
	tResult.mSize = tSigLen;
	tResult.mData = tSignature;

	return(tResult);
}

bool RSASignature::verifyMessage(const RSAKey& iPublicKey,
                                 DigestType iDigestType,
                                 const char* iMsg,
                                 size_t iMsgLen,
                                 const RSASignature& iSignature)
{
	if (!iPublicKey.isPublicKey())
		MIRA_THROW(XInvalidParameter,
		           "Need a public RSA key to verify a message.");

	if ((iMsg == NULL) || (iMsgLen == 0))
		MIRA_THROW(XInvalidParameter, "Can't sign an empty message.");

	///////////////////////////////////////////////////////////////////////////
	// build the digest of the message

	int tDigestType = 0;
	size_t tDigestLen = 0;
	unsigned char* tDigest = NULL;

	if (iDigestType == DIGEST_MD5) {
		tDigestType = NID_md5;
		tDigestLen = MD5_DIGEST_LENGTH;

		MD5_CTX tCtx;
		MD5_Init(&tCtx);
		MD5_Update(&tCtx, iMsg, iMsgLen);
		tDigest = new unsigned char[tDigestLen];
		MD5_Final(tDigest, &tCtx);
	} else
	if (iDigestType == DIGEST_SHA1) {
		tDigestType = NID_sha1;
		tDigestLen = SHA_DIGEST_LENGTH;

		SHA_CTX tCtx;
		SHA1_Init(&tCtx);
		SHA1_Update(&tCtx, iMsg, iMsgLen);
		tDigest = new unsigned char[tDigestLen];
		SHA1_Final(tDigest, &tCtx);
	} else
	if (iDigestType == DIGEST_RIPEMD160) {
		tDigestType = NID_ripemd160;
		tDigestLen = RIPEMD160_DIGEST_LENGTH;

		RIPEMD160_CTX tCtx;
		RIPEMD160_Init(&tCtx);
		RIPEMD160_Update(&tCtx, iMsg, iMsgLen);
		tDigest = new unsigned char[tDigestLen];
		RIPEMD160_Final(tDigest, &tCtx);
	}

	if (tDigest == NULL)
		MIRA_THROW(XInvalidParameter, "Unable to build message digest. "
		           "Invalid digest type?");

	///////////////////////////////////////////////////////////////////////////////
	// now verify the message

	if ((int)iSignature.mSize != RSA_size(iPublicKey.getOpenSSLKey()->key)) {
		delete [] tDigest;
		MIRA_THROW(XInvalidParameter, "Unexpected RSA signature length: "
		           "Size of the signature must be equal to the size of the key "
		           << "(KeySize=" << RSA_size(iPublicKey.getOpenSSLKey()->key)
		           << ", SignatureSize=" << iSignature.mSize << ").");
	}

	int tRes = RSA_verify(tDigestType, tDigest, tDigestLen, 
	                      iSignature.mData, iSignature.mSize,
	                      iPublicKey.getOpenSSLKey()->key);

	delete [] tDigest;

	return(tRes == 1);
}

///////////////////////////////////////////////////////////////////////////////

std::ostream& operator<<(std::ostream& stream, const RSASignature& signature)
{
	/**
	 * The following code doesn't work. The function BN_bn2hex will not 
	 * output leading zero(s), if signature.mData starts with one or more
	 * 0x00. Therefore, the size of resulting string would be too short.
	 */
	/*
	BIGNUM tBN;
	BN_init(&tBN);

	// convert signature data into a BIGNUM
	if (BN_bin2bn(signature.mData, signature.mSize, &tBN) == NULL)
		MIRA_THROW(XSystemCall, "Unable to convert signature to BIGNUM.");

	// convert BIGNUM into a hex string
	char* tStr = BN_bn2hex(&tBN);
	if (tStr == NULL)
		MIRA_THROW(XSystemCall, "Unable to convert BIGNUM to string.");

	// put hex string into stream
	stream << std::string(tStr);

	// cleanup and finish
	OPENSSL_free(tStr);
	return(stream);
	*/

	std::stringstream tStr;
	for(size_t i = 0; i < signature.mSize; i++)
		tStr << boost::format("%02X") % (int)signature.mData[i];

	stream << tStr.str();
	return(stream);
}

std::istream& operator>>(std::istream& stream, RSASignature& signature)
{
	// read hex string from stream
	std::string tStr;
	stream >> tStr;

	if ((tStr.size() % 2) != 0)
		MIRA_THROW(XInvalidParameter, "Need a even number of chars in "
		          "hex-string to convert to a RSA signature");

	/**
	 * The following code doesn't work. The function BN_hex2bn will not 
	 * convert leading zero(s), if tStr starts with one or more "00".
	 * Therefore, the size of resulting signature would be too short.
	 */
	/*
	// convert hex string into a BIGNUM
	BIGNUM* tBN = BN_new();
	int tBNLen = 0;
	if ((tBNLen = BN_hex2bn(&tBN, tStr.c_str())) == 0)
		MIRA_THROW(XSystemCall, "Unable to convert string to BIGNUM.");

	// convert BIGNUM into binary data
	int tLen = BN_num_bytes(tBN);
	uint8* tData = new uint8[tLen];
	if (BN_bn2bin(tBN, tData) != tLen)
		MIRA_THROW(XSystemCall, "Unable to convert BIGNUM into binary data.");

	// store into signature and cleanup
	delete [] signature.mData;
	signature.mData = tData;
	signature.mSize = tLen;
	BN_clear_free(tBN);
	*/

	size_t tLen = tStr.size()/2;
	uint8* tData = new uint8[tLen];
	for(size_t i = 0; i < tStr.size(); i += 2)
		tData[i/2] = (uint8)fromString<FromHex<uint16>>(tStr.substr(i, 2));

	// store into signature and cleanup
	delete [] signature.mData;
	signature.mData = tData;
	signature.mSize = tLen;

	return(stream);
}

///////////////////////////////////////////////////////////////////////////////

} // namespaces
