/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSON.C
 *    Implementation of JSON.h
 *
 * @author Erik Einhorn
 * @date   2011/12/20
 */

#include <json_spirit_utils.h>
#include <json_spirit_writer_template.h>
#include <json_spirit_reader_template.h>

#include <utils/StringAlgorithms.h>
#include <math/EigenFormat.h>

#include <json/JSON.h>

#ifndef MIRA_WINDOWS
template class json_spirit::Config_map<std::string>;
template class json_spirit::Value_impl<json_spirit::Config_map<std::string>>;
#endif

namespace mira { namespace json {

///////////////////////////////////////////////////////////////////////////////

void write(const Value& value, std::ostream& ioStream,
           bool formatted, int precision)
{
	int options = json_spirit::remove_trailing_zeros; // deprecated, no effect!
	if (formatted)
		// single_line_arrays implies pretty_print for json spirit currently, but better not rely on it
		options |= json_spirit::pretty_print | json_spirit::single_line_arrays;
	json_spirit::write_stream(value, ioStream, options,
	                          precision < 0 ? JSONDefaultPrecision::get() : precision);
}

std::string write(const Value& value, bool formatted, int precision)
{
	int options = json_spirit::remove_trailing_zeros; // deprecated, no effect!
	if (formatted)
		// single_line_arrays implies pretty_print for json spirit currently, but better not rely on it
		options |= json_spirit::pretty_print | json_spirit::single_line_arrays;
	return json_spirit::write_string(value, options,
	   	                             precision < 0 ? JSONDefaultPrecision::get() : precision);
}

void read(const std::string& s, Value& oValue)
{
	try
	{
		json_spirit::read_string_or_throw(s, oValue);
	}
	catch(std::string& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: " << ex);
	}
	catch(json_spirit::Error_position& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: On line "
		           << ex.line_ << " column " << ex.column_ << " reason " << ex.reason_);
	}
	catch(...)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string");
	}
}

void read(std::istream& ioStream, Value& oValue)
{
	try
	{
		json_spirit::read_stream_or_throw(ioStream, oValue);
	}
	catch(std::string& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: " << ex);
	}
	catch(json_spirit::Error_position& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: On line "
		           << ex.line_ << " column " << ex.column_ << " reason " << ex.reason_);
	}
	catch(...)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string");
	}
}

void read(std::string::const_iterator& begin,
          std::string::const_iterator& end, Value& oValue)
{
	try
	{
		json_spirit::read_range_or_throw(begin, end, oValue);
	}
	catch(std::string& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: " << ex);
	}
	catch(json_spirit::Error_position& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: On line "
		           << ex.line_ << " column " << ex.column_ << " reason " << ex.reason_);
	}
	catch(...)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string");
	}
}

Value getElement(const Value& iValue, const std::string& element)
{
	const Value* currentValue = &iValue;
	if (!element.empty())
	{
		std::vector<std::string> out;
		boost::split(out, element, boost::is_from_range('.','.'));
		for(std::size_t i=0; i<out.size(); ++i)
		{
			// check for array access
			std::size_t bBegin = out[i].find("[");
			if (bBegin != std::string::npos)
			{
				std::size_t bEnd = out[i].find("]", bBegin);
				if (bEnd == std::string::npos || (int)bEnd - (int)bBegin <= 1)
					MIRA_THROW(XInvalidConfig, "Array has wrong syntax "
					           "(check position and number of brackets, use [index])");
				if (bBegin > 0)
				{
					if (currentValue->type() != json_spirit::obj_type)
						MIRA_THROW(XInvalidConfig,
						           "Data type must be an object if member is specified");
					std::string objName = out[i].substr(0, bBegin);
					if (currentValue->get_obj().find(objName) == currentValue->get_obj().end())
						MIRA_THROW(XInvalidConfig, "Element " << element << " not found");
					currentValue = &currentValue->get_obj().at(objName);
				}
				std::size_t index = fromString<std::size_t>(out[i].substr(bBegin+1, 
				                                                          bEnd-bBegin-1));
				if (currentValue->type() == json_spirit::array_type)
				{
					if (index >= currentValue->get_array().size())
						MIRA_THROW(XInvalidConfig, "Array index is out of range");
					currentValue = &currentValue->get_array()[index];
				}
				else
					MIRA_THROW(XInvalidConfig, "Data type must be an array if index is specified");
				continue;
			}

			if (currentValue->type() != json_spirit::obj_type)
				MIRA_THROW(XInvalidConfig, "Data type must be an object if member is specified");
			if (currentValue->get_obj().find(out[i]) == currentValue->get_obj().end())
				MIRA_THROW(XInvalidConfig, "Member " << element << " not found");
			currentValue = &currentValue->get_obj().at(out[i]);
		}
	}
	return *currentValue;
}

double getNumberElement(const Value& iValue, const std::string& element)
{
	const Value* currentValue = &iValue;
	if (!element.empty())
	{
		std::vector<std::string> out;
		boost::split(out, element, boost::is_from_range('.','.'));
		for(std::size_t i=0; i<out.size(); ++i)
		{
			// check for array access
			std::size_t bBegin = out[i].find("[");
			if (bBegin != std::string::npos)
			{
				std::size_t bEnd = out[i].find("]", bBegin);
				if (bEnd == std::string::npos || (int)bEnd - (int)bBegin <= 1)
					MIRA_THROW(XInvalidConfig, "Array has wrong syntax "
					           "(check position and number of brackets, use [index])");
				if (bBegin > 0)
				{
					if (currentValue->type() != json_spirit::obj_type)
						MIRA_THROW(XInvalidConfig,
						           "Data type must be an object if member is specified");
					std::string objName = out[i].substr(0, bBegin);
					if (currentValue->get_obj().find(objName) == currentValue->get_obj().end())
						MIRA_THROW(XInvalidConfig, "Element " << element << " not found");
					currentValue = &currentValue->get_obj().at(objName);
				}
				std::size_t index = fromString<std::size_t>(out[i].substr(bBegin+1, 
				                                                          bEnd-bBegin-1));
				if (currentValue->type() == json_spirit::array_type)
				{
					if (index >= currentValue->get_array().size())
						MIRA_THROW(XInvalidConfig, "Array index is out of range");
					currentValue = &currentValue->get_array()[index];

					// check for valid multidim index syntax and give a helpful error message if necessary
					if (out[i].find("[", bEnd) != std::string::npos)
						MIRA_THROW(XInvalidConfig, "Matrix has wrong syntax "
						                           "(use [i].[j] for multiple dimension indices)");
				} else if (currentValue->type() == json_spirit::str_type)
				{
					if (i<out.size()-1)
						MIRA_THROW(XInvalidConfig, "Matrix element can not have a member");
					std::size_t b2Begin = out[i].find("[", bEnd+1);
					std::size_t b2End = out[i].find("]", bEnd+1);
					if (b2Begin == std::string::npos ||
						b2End == std::string::npos ||
						(int)b2End - (int)b2Begin <= 1)
						MIRA_THROW(XInvalidConfig, "Matrix has wrong syntax "
						           "(check position and number of brackets, use [row][col])");
					std::size_t index2 = fromString<std::size_t>(out[i].substr(b2Begin+1, 
					                                                           b2End-b2Begin-1));
					Eigen::MatrixXf m;
					std::stringstream ss(currentValue->get_str());
					ss >> format(m);
					if ((int)index >= m.rows() || (int)index2 >= m.cols())
						MIRA_THROW(XInvalidConfig, "Matrix index is out of range");
					return m(index, index2);
				}
				else
					MIRA_THROW(XInvalidConfig, "Data type must be an array or string "
					           "(matrix) if index is specified");
				continue;
			}

			if (currentValue->type() != json_spirit::obj_type)
				MIRA_THROW(XInvalidConfig, 
				           "Data type must be an object if member is specified");
			if (currentValue->get_obj().find(out[i]) == currentValue->get_obj().end())
				MIRA_THROW(XInvalidConfig, "Member " << element << " not found");
			currentValue = &currentValue->get_obj().at(out[i]);
		}
	}
	// special treatment for bool
	if (currentValue->type() == json_spirit::bool_type)
		return currentValue->get_bool()?1.0:0.0;
	if (currentValue->type() != json_spirit::int_type &&
		currentValue->type() != json_spirit::real_type)
		MIRA_THROW(XInvalidConfig, "Data must be of real or int type");
	return currentValue->get_value<double>();
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace
