/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDom.h
 *    A STL conform DOM reader/writer for XML.
 *
 * @author Tim Langner
 * @date   2010/07/01
 */

#ifndef _MIRA_XMLDOM_H_
#define _MIRA_XMLDOM_H_

#include <iostream>
#include <string>
#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#endif

#include <error/Exceptions.h>
#include <platform/Types.h>
#include <utils/Path.h>
#include <utils/ToString.h>

struct _xmlNode;
struct _xmlAttr;
struct _xmlDoc;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup XMLModule
 * A STL conform wrapper for libxml2 to read XML files as DOM
 */
class MIRA_BASE_EXPORT XMLDom : boost::noncopyable
{
private:
	enum
	{
		TEXT_NODE,
		COMMENT_NODE
	};
public:
	/** @name Public types. */
	//@{

	/// An XML attribute.
	typedef std::pair<std::string, std::string> Attribute;

	/// Namespace information of a node
	struct NameSpace
	{
		NameSpace() {}
		NameSpace(const std::string& p, const std::string& h="") :
			prefix(p),
			href(h) {}

		std::string prefix;
		std::string href;
	};

	//@}

public:
	/** @name Iterator types. */
	//@{

	// forward decl.
	class MIRA_BASE_EXPORT sibling_iterator;
	class MIRA_BASE_EXPORT const_sibling_iterator;

	/**
	 * @ingroup XMLModule
	 * Const iterator for iterating over attributes
	 */
	class MIRA_BASE_EXPORT const_attribute_iterator
	{
	public:
		/**
		 * STL-conform typedefs
		 */
		/// The value type.
		typedef std::string value_type;
		/// The pointer type.
		typedef const std::string* pointer;
		/// The reference type.
		typedef const std::string& reference;
		/// The difference type.
		typedef ptrdiff_t difference_type;
		/// The iterator category.
		typedef std::bidirectional_iterator_tag iterator_category;

		/**
		 * Constructor
		 */
		const_attribute_iterator();

		/**
		 * Constructor
		 * @param[in] iNode the parent node for the attribute
		 * @param[in] iAttribute the attribute
		 */
		const_attribute_iterator(_xmlNode* iNode, _xmlAttr* iAttribute);

		/**
		 * Compare two attribute iterators
		 * @param[in] other The other iterator
		 * @return true if equal
		 */
		bool operator==(const const_attribute_iterator& other) const;
		/**
		 * Compare two attribute iterators
		 * @param[in] other The other iterator
		 * @return true if not equal
		 */
		bool operator!=(const const_attribute_iterator& other) const;
		/**
		 * Preincrement
		 */
		const_attribute_iterator& operator++();
		/**
		 * Predecrement
		 */
		const_attribute_iterator& operator--();

		/**
		 * Dereference operator Attribute.
		 * @return A pair of attribute name and value attribute.
		 */
		Attribute operator*() const;
		/**
		 * Get the name of the attribute.
		 * @return The name.
		 */
		std::string name() const;
		/**
		 * Get the value of the attribute.
		 * @return The value.
		 */
		std::string value() const;
		/**
		 * Return the namespace information of the attribute
		 * @return Namespace information
		 */
		NameSpace nameSpace() const;

	private:
		/// The node pointer
		_xmlNode* mNode;
		/// The attribute pointer
		_xmlAttr* mAttribute;
	};

	/**
	 * @ingroup XMLModule
	 * Iterator for iterating over attributes
	 */
	class MIRA_BASE_EXPORT attribute_iterator
	{
	public:
		/**
		 * STL-conform typedefs
		 */
		/// The value type.
		typedef std::string value_type;
		/// The pointer type.
		typedef std::string* pointer;
		/// The reference type.
		typedef std::string& reference;
		/// The difference type.
		typedef ptrdiff_t difference_type;
		/// The iterator category.
		typedef std::bidirectional_iterator_tag iterator_category;

		/**
		 * Constructor
		 */
		attribute_iterator();

		/**
		 * Constructor
		 * @param[in] iNode the parent node for the attribute
		 * @param[in] iAttribute the attribute
		 */
		attribute_iterator(_xmlNode* iNode, _xmlAttr* iAttribute);

		/**
		 * Set the value of the attribute.
		 * @param[in] value The value.
		 */
		const attribute_iterator& operator=(const std::string& value);
		/**
		 * Compare two attribute iterators
		 * @param[in] other The other iterator
		 * @return true if equal
		 */
		bool operator==(const attribute_iterator& other) const;
		/**
		 * Compare two attribute iterators
		 * @param[in] other The other iterator
		 * @return true if not equal
		 */
		bool operator!=(const attribute_iterator& other) const;
		/**
		 * Preincrement
		 */
		attribute_iterator& operator++();
		/**
		 * Predecrement
		 */
		attribute_iterator& operator--();

		/**
		 * Dereference operator to Attribute.
		 * @return A pair of attribute name and value attribute.
		 */
		Attribute operator*() const;
		/**
		 * Get the name of the attribute.
		 * @return The name.
		 */
		std::string name() const;
		/**
		 * Get the value of the attribute.
		 * @return The value.
		 */
		std::string value() const;
		/**
		 * Return the namespace information of the node
		 * @return Namespace information
		 */
		NameSpace nameSpace() const;

	private:
		friend class sibling_iterator; // sibling_iterator needs direct access on in remove

		/// The node pointer
		_xmlNode* mNode;
		/// The attribute pointer
		_xmlAttr* mAttribute;
	};

	/**
	 * @ingroup XMLModule
	 * Const iterator for iterating over data nodes
	 */
	template <int Type>
	class MIRA_BASE_EXPORT const_data_iterator
	{
	public:
		/**
		 * STL-conform typedefs
		 */
		/// The value type.
		typedef std::string value_type;
		/// The pointer type.
		typedef const std::string* pointer;
		/// The reference type.
		typedef const std::string& reference;
		/// The difference type.
		typedef ptrdiff_t difference_type;
		/// The iterator category.
		typedef std::bidirectional_iterator_tag iterator_category;

		/**
		 * Constructor
		 */
		const_data_iterator();

		/**
		 * Constructor
		 * @param[in] iNode The current node
		 */
		const_data_iterator(_xmlNode* iNode);

		/**
		 * Compare two iterators
		 * @param[in] other The other iterator
		 * @return true if equal
		 */
		bool operator==(const const_data_iterator& other) const;
		/**
		 * Compare two iterators
		 * @param[in] other The other iterator
		 * @return true if not equal
		 */
		bool operator!=(const const_data_iterator& other) const;
		/**
		 * Preincrement
		 */
		const_data_iterator& operator++();
		/**
		 * Predecrement
		 */
		const_data_iterator& operator--();

		/**
		 * Dereference operator to std::string.
		 * @return Returns the data value.
		 */
		std::string operator*() const;

	private:

		friend class sibling_iterator;
		/// The node pointer
		_xmlNode* mNode;
	};

	/**
	 * Iterator for iterating over data nodes
	 */
	template <int Type>
	class MIRA_BASE_EXPORT data_iterator
	{
	public:
		/**
		 * STL-conform typedefs
		 */
		/// The value type.
		typedef std::string value_type;
		/// The pointer type.
		typedef std::string* pointer;
		/// The reference type.
		typedef std::string& reference;
		/// The difference type.
		typedef ptrdiff_t difference_type;
		/// The iterator category.
		typedef std::bidirectional_iterator_tag iterator_category;


		/**
		 * Constructor
		 */
		data_iterator();

		/**
		 * Constructor
		 * @param[in] iNode The current node.
		 */
		data_iterator(_xmlNode* iNode);

		/**
		 * Set the data value of the node.
		 * @param[in] value The value.
		 */
		const data_iterator& operator=(const std::string& value);
		/**
		 * Compare two iterators
		 * @param[in] other The other iterator
		 * @return true if equal
		 */
		bool operator==(const data_iterator& other) const;
		/**
		 * Compare two iterators
		 * @param[in] other The other iterator
		 * @return true if not equal
		 */
		bool operator!=(const data_iterator& other) const;
		/**
		 * Preincrement
		 */
		data_iterator& operator++();
		/**
		 * Predecrement
		 */
		data_iterator& operator--();

		/**
		 * Dereference operator to std::string.
		 * @return Returns the data value.
		 */
		std::string operator*() const;

		/**
		 * Removes the data node the iterator points to.
		 * @return The iterator pointing to the next data node after the deleted one
		 */
		data_iterator remove();

	private:
		/// The node pointer
		_xmlNode* mNode;
	};

	/**
	 * STL-conform typedefs
	 */
	/**
	 * @ingroup XMLModule
	 * The const iterator for comments.
	 */
	typedef const_data_iterator<COMMENT_NODE> const_comment_iterator;
	/**
	 * @ingroup XMLModule
	 * The const iterator for content.
	 */
	typedef const_data_iterator<TEXT_NODE> const_content_iterator;
	/**
	 * @ingroup XMLModule
	 * The iterator for comments.
	 */
	typedef data_iterator<COMMENT_NODE> comment_iterator;
	/**
	 * @ingroup XMLModule
	 * The iterator for content.
	 */
	typedef data_iterator<TEXT_NODE> content_iterator;

	/**
	 * Base iterator for iterating over xml nodes with the
	 * same parent (siblings)
	 */
	class MIRA_BASE_EXPORT iterator_base
	{
	public:
		/**
		 * Constructor
		 * @param[in] node The actual node.
		 * @param[in] name For iterating over siblings with same name
		 */
		iterator_base(_xmlNode* node, const std::string& name);
		/**
		 * Dereference operator to get the name of the node.
		 * @return Returns the node name.
		 */
		std::string operator*() const;

		/**
		 * Change the node's name.
		 */
		void setName(const std::string& name);

		/**
		 * Return the uri of the node. This can be the filename of the document 
		 * or a web address.
		 * @return The uri of the node.
		 */
		std::string uri() const;

		/**
		 * Set the uri of the node. This can be the filename of the document
		 * or a web address.
		 */
		void setUri(const std::string& uri);

		/**
		 * Return the namespace information of the node
		 * @return Namespace information
		 */
		NameSpace nameSpace() const;

		/**
		 * Gets the underlying line number of the loaded document for this node.
		 * @return Line number
		 */
		uint32 line() const;

		/**
		 * Get the const parent node sibling_iterator for this node
		 * @return Const sibling_iterator to parent node
		 */
		const_sibling_iterator cparent() const;
		/**
		 * Get the const parent node sibling_iterator for this node
		 * @return Const sibling_iterator to parent node
		 */
		const_sibling_iterator parent() const { return cparent(); }
		/**
		 * Get the const sibling_iterator to the first sub node.
		 * @return The const sibling_iterator.
		 */
		const_sibling_iterator cbegin() const;
		/**
		 * Get the const sibling_iterator to the first sub node.
		 * @return The const sibling_iterator.
		 */
		const_sibling_iterator begin() const { return cbegin(); }
		/**
		 * Get the const sibling_same_name_iterator to the first sub node
		 * with the given name.
		 * @param name The name of the sibling
		 * @return The const sibling_same_name_iterator.
		 */
		const_sibling_iterator cbegin(const std::string& name) const;
		/**
		 * Get the const sibling_same_name_iterator to the first sub node
		 * with the given name.
		 * @param name The name of the sibling
		 * @return The const sibling_same_name_iterator.
		 */
		const_sibling_iterator begin(const std::string& name) const {
			return cbegin(name);
		}
		/**
		 * Get the const end node sibling_iterator
		 * @return The const end sibling_iterator.
		 */
		const_sibling_iterator cend() const;
		/**
		 * Get the const end node sibling_iterator
		 * @return The const end sibling_iterator.
		 */
		const_sibling_iterator end() const { return cend(); }
		/**
		 * Get the const iterator to the first content.
		 * @return The const content iterator.
		 */
		const_content_iterator content_cbegin() const;
		/**
		 * Get the const end content iterator.
		 * @return The const end iterator.
		 */
		const_content_iterator content_cend() const;
		/**
		 * Get the const iterator to the first content.
		 * @return The const content iterator.
		 */
		const_content_iterator content_begin() const { return content_cbegin(); }
		/**
		 * Get the const end content iterator
		 * @return The const end iterator.
		 */
		const_content_iterator content_end() const { return content_cend(); }

		/**
		 * Get the const iterator to the first comment.
		 * @return The const comment iterator.
		 */
		const_comment_iterator comment_cbegin() const;
		/**
		 * Get the const end comment iterator
		 * @return The const end iterator.
		 */
		const_comment_iterator comment_cend() const;
		/**
		 * Get the const iterator to the first comment.
		 * @return The const comment iterator.
		 */
		const_comment_iterator comment_begin() const { return comment_cbegin(); }
		/**
		 * Get the const end comment iterator
		 * @return The const end iterator.
		 */
		const_comment_iterator comment_end() const { return comment_cend(); }

		/**
		 * Get the const iterator to the first attribute.
		 * @return The const attribute iterator.
		 */
		const_attribute_iterator attribute_cbegin() const;
		/**
		 * Get the const end attribute iterator
		 * @return The const end iterator.
		 */
		const_attribute_iterator attribute_cend() const;
		/**
		 * Get the const iterator to the first attribute.
		 * @return The const attribute iterator.
		 */
		const_attribute_iterator attribute_begin() const 
		{
			return attribute_cbegin(); 
		}
		/**
		 * Get the const end attribute iterator
		 * @return The const end iterator.
		 */
		const_attribute_iterator attribute_end() const 
		{
			return attribute_cend(); 
		}
		/**
		 * Find an attribute with the name name.
		 * @param[in] name The attribute name.
		 * @return The const attribute iterator. end iterator if not exists.
		 */
		const_attribute_iterator find_attribute(const std::string& name) const;
		/**
		 * Returns true if node contains the attribute with name name.
		 * @param[in] name The attribute name.
		 * @return true if the attribute exists.
		 */
		bool has_attribute(const std::string& name) const
		{
			return find_attribute(name) != attribute_end();
		}
		/**
		 * Return an attribute value cast to a given type
		 * @throws XInvalidConfig when the attribute could not be found
		 * @param[in] name The name of the attribute
		 * @return The casted value of the attribute
		 */
		template <typename T>
		T get_attribute(const std::string& name) const
		{
			const_attribute_iterator a = find_attribute(name);
			if ( a == attribute_end() )
				MIRA_THROW(XInvalidConfig, "The attribute '" << name
				           << "' could not be found in node '" << *(*this) << "'");
			return fromString<T>((*a).second);
		}

		/**
		 * Return an attribute value casted to a given type
		 * If the attribute could not be found the given default value will be returned
		 * @param[in] name The name of the attribute
		 * @param[in] defaultValue The default value that will be returned if 
		 * the attribute is not found
		 * @return The casted value of the attribute
		 */
		template <typename T>
		T get_attribute(const std::string& name, const T& defaultValue) const
		{
			const_attribute_iterator a = find_attribute(name);
			if ( a == attribute_end() )
				return defaultValue;
			return fromString<T>((*a).second);
		}

	protected:
		/// The node pointer
		_xmlNode* mNode;
		/// the node name (optional)
		std::string mName;
	};

	/**
	 * @ingroup XMLModule
	 * Const sibling_iterator for iterating over xml nodes
	 * that have the same parent (siblings)
	 */
	class MIRA_BASE_EXPORT const_sibling_iterator : public iterator_base
	{
	public:
		/**
		 * STL-conform typedefs
		 */
		friend class sibling_iterator;
		/// The value type.
		typedef std::string value_type;
		/// The pointer type.
		typedef const std::string* pointer;
		/// The reference type.
		typedef const std::string& reference;
		/// The difference type.
		typedef ptrdiff_t difference_type;
		/// The iterator category.
		typedef std::bidirectional_iterator_tag iterator_category;

		/**
		 * Constructor
		 */
		const_sibling_iterator();
		/**
		 * Constructor
		 * @param[in] iNode The actual node.
		 */
		const_sibling_iterator(_xmlNode* iNode, const std::string& name = "");

		/**
		 * Copy-constructor
		 * @param[in] other The other const_sibling_iterator
		 */
		const_sibling_iterator(const const_sibling_iterator& other) :
			iterator_base(other.mNode, other.mName)
		{
		}

		/**
		 * Compare two iterators
		 * @param[in] other The other sibling_iterator
		 * @return true if equal
		 */
		bool operator==(const const_sibling_iterator& other) const {
			return mNode == other.mNode;
		}
		/**
		 * Compare two iterators
		 * @param[in] other The other sibling_iterator
		 * @return true if not equal
		 */
		bool operator!=(const const_sibling_iterator& other) const {
			return mNode != other.mNode;
		}
		/**
		 * Preincrement
		 */
		const_sibling_iterator& operator++();
		/**
		 * Increment self by given value
		 */
		const_sibling_iterator& operator+=(std::size_t increment);
		/**
		 * Increment by given value
		 */
		const_sibling_iterator operator+(std::size_t increment) const;
		/**
		 * Predecrement
		 */
		const_sibling_iterator& operator--();

		/**
		 * Find the nth subnode with a given name.
		 * The name can contain '/''s for separation of child names. In this case
		 * the DOM tree is traversed recursively. e.g. find("Node/SubNode/SubSubNode")
		 * returns an sibling_iterator to SubSubNode if exists. It returns the
		 * nth SubSubNode of the first SubNode in the first Node.)
		 * @param[in] name The name of the node
		 * @param[in] nth The nth item with name will be found
		 * @return Const sibling_iterator to found nth node or end sibling_iterator if not found
		 */
		const_sibling_iterator find(const std::string& name, std::size_t nth = 0) const;
	};

	/**
	 * @ingroup XMLModule
	 * Iterator for iterating over xml nodes that have the same parent (sibligs)
	 */
	class MIRA_BASE_EXPORT sibling_iterator : public iterator_base
	{
	public:
		/**
		 * STL-conform typedefs
		 */
		/// The value type.
		typedef std::string value_type;
		/// The pointer type.
		typedef std::string* pointer;
		/// The reference type.
		typedef std::string& reference;
		/// The difference type.
		typedef ptrdiff_t difference_type;
		/// The iterator category.
		typedef std::bidirectional_iterator_tag iterator_category;

		/**
		 * Default-constructor
		 */
		sibling_iterator();
		/**
		 * Constructor
		 * @param[in] iNode The actual node.
		 */
		sibling_iterator(_xmlNode* iNode, const std::string& name = "");

		/**
		 * Copy-constructor
		 * @param[in] other The other sibling_iterator
		 */
		sibling_iterator(const sibling_iterator& other) :
			iterator_base(other.mNode, other.mName)
		{
		}

		/**
		 * Compare two iterators
		 * @param[in] other The other sibling_iterator
		 * @return true if equal
		 */
		bool operator==(const sibling_iterator& other) const {
			return mNode == other.mNode;
		}
		/**
		 * Compare two iterators
		 * @param[in] other The other sibling_iterator
		 * @return true if not equal
		 */
		bool operator!=(const sibling_iterator& other) const {
			return mNode != other.mNode;
		}

		/**
		 * Converts this sibling_iterator to a const sibling_iterator
		 */
		operator const_sibling_iterator() const
		{
			return const_sibling_iterator(mNode, mName);
		}

		/**
		 * Change the name of this node.
		 * @param[in] name The new name.
		 * @return This sibling_iterator.
		 */
		const sibling_iterator& operator=(const std::string& name);
		/**
		 * Preincrement
		 */
		sibling_iterator& operator++();
		/**
		 * Increment self by given value
		 */
		sibling_iterator& operator+=(std::size_t increment);
		/**
		 * Increment by given value
		 */
		sibling_iterator operator+(std::size_t increment) const;
		/**
		 * Predecrement
		 */
		sibling_iterator& operator--();

		/**
		 * Get the parent node sibling_iterator for this node
		 * @return Iterator to parent node
		 */
		sibling_iterator parent();
		/**
		 * Get the sibling_iterator to the first sub node.
		 * @return The sibling_iterator.
		 */
		sibling_iterator begin();
		/**
		 * Get the sibling_iterator to the first sub node with given name.
		 * If the returned iterator is incremented or decremented it will move
		 * to the next/previous sibling with the same name.
		 * @param[in] name The name of the sibling
		 * @return The sibling_iterator.
		 */
		sibling_iterator begin(const std::string& name);
		/**
		 * Get the end node sibling_iterator
		 * @return The end sibling_iterator.
		 */
		sibling_iterator end();

		/**
		 * Find the nth subnode with a given name.
		 * The name can contain '/''s for separation of child names. In this case
		 * the DOM tree is traversed recursively. e.g. find("Node/SubNode/SubSubNode")
		 * returns an sibling_iterator to SubSubNode if exists. It returns the
		 * nth SubSubNode of the first SubNode in the first Node.)
		 * @param[in] name The name of the node
		 * @param[in] nth The nth item with name will be found
		 * @return Iterator to found nth node or end sibling_iterator if not found
		 */
		sibling_iterator find(const std::string& name, std::size_t nth = 0);

		/**
		 * Get the iterator to the first content.
		 * @return The content iterator.
		 */
		content_iterator content_begin();
		/**
		 * Get the end content iterator
		 * @return The end iterator.
		 */
		content_iterator content_end();
		/**
		 * Get the iterator to the first comment.
		 * @return The comment iterator.
		 */
		comment_iterator comment_begin();
		/**
		 * Get the end comment sibling_iterator
		 * @return The end sibling_iterator.
		 */
		comment_iterator comment_end();
		/**
		 * Get the iterator to the first attribute.
		 * @return The attribute iterator.
		 */
		attribute_iterator attribute_begin();
		/**
		 * Get the end attribute iterator
		 * @return The end iterator.
		 */
		attribute_iterator attribute_end();
		/**
		 * Find an attribute with the name name.
		 * @param[in] name The attribute name.
		 * @return The attribute iterator. end iterator if not exists.
		 */
		attribute_iterator find_attribute(const std::string& name);
		/**
		 * Add an attribute to this node.
		 * @param[in] attribute The attribute pair.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& add_attribute(const Attribute& attribute);
		/**
		 * Add an attribute to this node.
		 * @param[in] name The attribute name.
		 * @param[in] value The attribute value.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& add_attribute(const std::string& name,
		                                const std::string& value);
		/**
		 * Add an attribute to this node.
		 * @param[in] name The attribute name.
		 * @param[in] value The attribute value.
		 * @return This sibling_iterator.
		 */
		template <typename T>
		XMLDom::sibling_iterator& add_attribute(const std::string& name, const T& value)
		{
			return add_attribute(name, toString(value));
		}

		/**
		 * Add an attribute to this node.
		 * @param[in] name The attribute name.
		 * @param[in] value The attribute value.
		 * @param[in] precision The precision that should be used, e.g. if the 
		 * attribute is a floating point number.
		 * @return This sibling_iterator.
		 */
		template <typename T>
		XMLDom::sibling_iterator& add_attribute(const std::string& name, const T& value,
		                                int precision)
		{
			return add_attribute(name, toString(value,precision));
		}

		/**
		 * Removes the attribute the given iterator points on from this node,
		 * if it exists. Otherwise, does nothing.
		 * The iterator will become invalid after this operation.
		 */
		void remove_attribute(attribute_iterator it);

		/**
		 * Add a comment to this node.
		 * @param[in] comment The comment string.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& add_comment(const std::string& comment);
		/**
		 * Add content to this node.
		 * @param[in] content The content string.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& add_content(const std::string& content);

		/**
		 * Add a cdata content block to this node.
		 * @param[in] cdata_content The content string that will be wrapped in cdata tags.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& add_cdata_content(const std::string& cdata_content);

		/**
		 * Add a sub node to this node.
		 * @param[in] name The name of the sub node.
		 * @return The sibling_iterator to the new sub node.
		 */
		XMLDom::sibling_iterator add_child(const std::string& name, const NameSpace& ns = NameSpace());
		/**
		 * Add a sub node to this node.
		 * @param[in] node The child node to add.
		 * @return The sibling_iterator to the new sub node.
		 */
		XMLDom::sibling_iterator add_child(const XMLDom::const_sibling_iterator& node);
		/**
		 * Inserts a given node before this node.
		 * @param[in] node The node to be inserted
		 * @return Iterator to the inserted node
		 */
		XMLDom::sibling_iterator insert_before(const XMLDom::const_sibling_iterator& node);
		/**
		 * Inserts a sub node before this node.
		 * @param[in] name The name of the sub node.
		 * @return The sibling_iterator to the new sub node.
		 */
		XMLDom::sibling_iterator insert_before(const std::string& name, const NameSpace& ns = NameSpace());
		/**
		 * Inserts a given node after this node.
		 * @param[in] node The node to be inserted
		 * @return Iterator to the inserted node
		 */
		XMLDom::sibling_iterator insert_after(const XMLDom::const_sibling_iterator& node);
		/**
		 * Inserts a sub node after this node.
		 * @param[in] name The name of the sub node.
		 * @return The sibling_iterator to the new sub node.
		 */
		XMLDom::sibling_iterator insert_after(const std::string& name, const NameSpace& ns = NameSpace());
		/**
		 * Replaces this node by a recursive copy of iNode.
		 * @param[in] node The node to copy
		 * @return The sibling_iterator pointing to the replaced node
		 */
		XMLDom::sibling_iterator replace(const XMLDom::const_sibling_iterator& node);
		/**
		 * Replaces this node by new one
		 * @param[in] name The name of the new node.
		 * @return The sibling_iterator pointing to the replaced node
		 */
		XMLDom::sibling_iterator replace(const std::string& name, const NameSpace& ns = NameSpace());
		/**
		 * Insert a comment before this node.
		 * @param[in] comment The comment node to insert.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_comment_before(const XMLDom::const_comment_iterator& comment);
		/**
		 * Insert a comment before this node.
		 * @param[in] comment The comment string.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_comment_before(const std::string& comment);
		/**
		 * Insert a comment after this node.
		 * @param[in] comment The comment node to insert.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_comment_after(const XMLDom::const_comment_iterator& comment);
		/**
		 * Insert a comment after this node.
		 * @param[in] comment The comment string.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_comment_after(const std::string& comment);
		/**
		 * Insert content before this node.
		 * @param[in] content The content node to insert.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_content_before(const XMLDom::const_content_iterator& content);
		/**
		 * Insert content before this node.
		 * @param[in] content The content string.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_content_before(const std::string& content);
		/**
		 * Insert content after this node.
		 * @param[in] content The content node to insert.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_content_after(const XMLDom::const_content_iterator& content);
		/**
		 * Insert content after this node.
		 * @param[in] content The content string.
		 * @return This sibling_iterator.
		 */
		XMLDom::sibling_iterator& insert_content_after(const std::string& content);
		/**
		 * Replaces this node by comment node.
		 * @param[in] node The comment node to copy
		 * @return The sibling_iterator pointing to the next node
		 */
		XMLDom::sibling_iterator replace_by_comment(const XMLDom::const_comment_iterator& node);
		/**
		 * Replaces this node by a comment
		 * @param[in] comment The comment.
		 * @return The sibling_iterator pointing to the next node
		 */
		XMLDom::sibling_iterator replace_by_comment(const std::string& comment);
		/**
		 * Replaces this node by content node.
		 * @param[in] node The content node to copy
		 * @return The sibling_iterator pointing to the next node
		 */
		XMLDom::sibling_iterator replace_by_content(const XMLDom::const_content_iterator& node);
		/**
		 * Replaces this node by content
		 * @param[in] content The content.
		 * @return The sibling_iterator pointing to the next node
		 */
		XMLDom::sibling_iterator replace_by_content(const std::string& content);
		/**
		 * Removes this node from the document.
		 * @return The sibling_iterator pointing to the next node after the deleted one
		 */
		XMLDom::sibling_iterator remove();
	};

	//@}

	/// typedefs for backward compatibility
	typedef sibling_iterator iterator;
	typedef const_sibling_iterator const_iterator;

public:
	/** @name Constructor and destructor. */
	//@{

	/**
	 * Constructor for creating a new empty document. The name of the root
	 * node can specified as optional parameter. The default name is "root".
	 */
	XMLDom(const std::string& rootNodeName = "root");

	/// Destructor
	~XMLDom();

	/// Move constructor
	XMLDom(XMLDom&& other);

	/// Move assignment operator
	XMLDom& operator=(XMLDom&& other);

	//@}

public:

	/**
	 * Clears the whole content of this XMLDom document, i.e. all nodes
	 * are destroyed and the XMLDom object has the same state as after calling
	 * it's constructor.
	 * Note: All iterators become invalid by calling this method. Accessing
	 * the iterators after calling this method will result in undefined
	 * behavior.
	 */
	void clear();

public:
	/** @name I/O operations. */
	//@{

	/**
	 * @brief Load and parse an XML document from memory.
	 * @throw XIO if the document could not be loaded.
	 * @param[in] buffer The buffer.
	 */
	void loadFromString(const std::string& buffer);
	/**
	 * @brief Load and parse an XML document from a file.
	 * @throw XFileNotFound if file is not found.
	 * @throw XIO if the file/document could not be loaded.
	 * @param[in] filename The file name.
	 */
	void loadFromFile(const Path& filename);

	/**
	 * @brief Save the XML document to a file.
	 * @throw XInvalidConfig if the document is empty.
	 * @throw XIO if the file could not be saved.
	 * @param[in] filename The file name.
	 * @param[in] encoding The desired encoding.
	 */
	void saveToFile(const Path& filename,
		const std::string& encoding = "UTF-8") const;

	/**
	 * @brief Save the XML document to a string.
	 * @throw XInvalidConfig if the document is empty.
	 * @throw XIO if the file could not be saved.
	 * @param[in] encoding The desired encoding.
	 * @return The string containing the xml document
	 */
	std::string saveToString(const std::string& encoding = "UTF-8") const;

	//@}

public:
	/** @name Iterators. */
	//@{

	/// Return a const sibling_iterator to the root node of the XML document.
	const_sibling_iterator croot() const;

	/// Return a const sibling_iterator to the root node of the XML document.
	const_sibling_iterator root() const { return croot(); }

	/// Return a sibling_iterator to the root node of the XML document.
	sibling_iterator root();

	//@}

	/**
	 * Get the URL of the document (i.e. the filename if loaded from file)
	 * @return URL of the document
	 */
	std::string uri() const;
	/**
	 * Set the URL of the document (i.e. the filename if loaded from file)
	 */
	void setUri(const std::string& uri);

	/**
	 * Get the initial encoding of the document
	 * @return Encoding if any.
	 */
	std::string encoding() const;

private:
	_xmlDoc* mXMLDocument;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
