/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UniqueIdentifier.h
 *    Determine a unique identifier (or string).
 *
 * @author Christof Schröter
 * @date   2018/08/29
 */

#ifndef _MIRA_UNIQUE_IDENTIFIER_H_
#define _MIRA_UNIQUE_IDENTIFIER_H_

#ifndef _FILE_NAME_
	#pragma message("Macro _FILE_NAME_ not defined (it should be an explicit " \
                        "definition in cmake). Disabling MIRA_UNIQUE_IDENTIFIER.")
#else

#include <boost/preprocessor/cat.hpp>
#include <boost/preprocessor/stringize.hpp>

#define MIRA_UNIQUE_IDENTIFIER_ENABLED

/**
 * Resolves to a unique valid identifier of the form __project_path_to_file_line_counter__ .
 *
 * \note: There are theoretical edge cases where this would be not unique, like compiling
 * the same source file multiple times (into one or multiple targets), or loading libs
 * from different projects with partially identical file structure.
 *
 * _FILE_NAME_ must have been defined appropriately by the make system,
 * __LINE__ and __COUNTER__ are assumed to be supported by the compiler.
 */ 
#define MIRA_UNIQUE_IDENTIFIER                                                 \
BOOST_PP_CAT(BOOST_PP_CAT(BOOST_PP_CAT(BOOST_PP_CAT(BOOST_PP_CAT(BOOST_PP_CAT( \
	__,_FILE_NAME_),_),__LINE__),_),__COUNTER__),__)

/**
 * Resolves to a semi-unique valid identifier of the form __project_path_to_file_line__ .
 * In contrast to MIRA_UNIQUE_IDENTIFIER, this value will not change when called
 * multiple times on the same source code line. That means e.g. it will hold
 * the same ("unique") value throughout a macro body.
 * \note: The value will also be identical if used for multiple arguments in one call
 * to a macro, even if those appear on different lines, e.g. both these examples are
 * equivalent and will provide 3 identical arguments:
 *
 * \code
 * MY_MACRO(MIRA_FILE_LINE_IDENTIFIER, MIRA_FILE_LINE_IDENTIFIER, MIRA_FILE_LINE_IDENTIFIER)
 *
 * MY_MACRO(MIRA_FILE_LINE_IDENTIFIER,
 *          MIRA_FILE_LINE_IDENTIFIER,
 *          MIRA_FILE_LINE_IDENTIFIER)
 * \endcode
 *
 * But not this one:
 *
 * \code
 * my_function(MIRA_FILE_LINE_IDENTIFIER,
 *             MIRA_FILE_LINE_IDENTIFIER,
 *             MIRA_FILE_LINE_IDENTIFIER);
 * \endcode
 * 
 * _FILE_NAME_ must have been defined appropriately by the make system,
 * __LINE__ is assumed to be supported by the compiler.
 */ 
#define MIRA_FILE_LINE_IDENTIFIER                                              \
BOOST_PP_CAT(BOOST_PP_CAT(BOOST_PP_CAT(BOOST_PP_CAT(__,_FILE_NAME_),_),__LINE__),__)

/**
 * Resolves to a unique string literal similar to MIRA_UNIQUE_IDENTIFIER.
 */ 
#define MIRA_UNIQUE_STRING \
BOOST_PP_STRINGIZE(MIRA_UNIQUE_IDENTIFIER)

/**
 * Resolves to a semi-unique string literal similar to MIRA_FILE_LINE_IDENTIFIER.
 */ 
#define MIRA_FILE_LINE_STRING \
BOOST_PP_STRINGIZE(MIRA_FILE_LINE_IDENTIFIER)

#endif // #ifndef _FILE_NAME_

#endif
