/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Pose.h
 *    Typedefs for different Pose datatypes that are internally RigidTransforms.
 *
 * @author Tim Langner
 * @date   2010/08/02
 */

#ifndef _MIRA_POSE_H_
#define _MIRA_POSE_H_

#include <utils/Stamped.h>

#include <transform/RigidTransform.h>

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

// some typedefs for easier access

/// A 2D pose consisting of a translation and rotation  @ingroup TransformModule
typedef RigidTransform<float,2> Pose2;

/// A 3D pose consisting of a translation and rotation @ingroup TransformModule
typedef RigidTransform<float,3> Pose3;

/// A 2D pose with covariance consisting of a translation and rotation @ingroup TransformModule
typedef RigidTransformCov<float,2> PoseCov2;

/// A 3D pose with covariance consisting of a translation and rotation @ingroup TransformModule
typedef RigidTransformCov<float,3> PoseCov3;

///////////////////////////////////////////////////////////////////////////////

/**
 * Typetrait that returns the correct vector type, that depends on whether
 * the Eigen-datatypes of the pose need an alignment or not.
 */
template <typename TPose>
struct PoseVectorTrait {
	typedef std::vector<TPose> VectorType;
	static const bool needsAlignment = false;
};

template <>
struct PoseVectorTrait<Pose3> {
	typedef std::vector<Pose3, Eigen::aligned_allocator<Pose3>> VectorType;
	static const bool needsAlignment = true;
};

template <>
struct PoseVectorTrait<PoseCov3> {
	typedef std::vector<PoseCov3, Eigen::aligned_allocator<PoseCov3>> VectorType;
	static const bool needsAlignment = true;
};

///////////////////////////////////////////////////////////////////////////////

/// Vector of Pose2 @ingroup TransformModule
typedef PoseVectorTrait<Pose2>::VectorType Pose2Vector;

/// Vector of Pose3 @ingroup TransformModule
typedef PoseVectorTrait<Pose3>::VectorType Pose3Vector;

/// Vector of PoseCov2 @ingroup TransformModule
typedef PoseVectorTrait<PoseCov2>::VectorType PoseCov2Vector;

/// Vector of PoseCov3 @ingroup TransformModule
typedef PoseVectorTrait<PoseCov3>::VectorType PoseCov3Vector;



/// deprecated, use PoseCov2Vector instead
typedef PoseCov2Vector Pose2CovVector;

/// deprecated, use PoseCov3Vector instead
typedef PoseCov3Vector Pose3CovVector;


///////////////////////////////////////////////////////////////////////////////

}

#endif
