/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file optional.hpp
 *    Reflect for boost::optional.
 *
 * @author Tim Langner
 * @date   2011/06/14
 */

#ifndef _MIRA_OPTIONAL_HPP_
#define _MIRA_OPTIONAL_HPP_

#ifndef Q_MOC_RUN
#include <boost/optional.hpp>
#endif

#include <serialization/ReflectorInterface.h>
#include <serialization/SplitReflect.h>

#include <serialization/IsTransparentSerializable.h>
#include <serialization/IsObjectTrackable.h>
#include <serialization/IsNotMetaSerializable.h>

#include <serialization/BinarySerializer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Specialization of the non-intrusive reflect for boost::optional.
template<typename DerivedReflector, typename T>
void reflectRead(ReflectorInterface<DerivedReflector>& r, boost::optional<T>& t)
{
	if (t) {
		DerivedReflector& reflector = static_cast<DerivedReflector&>(r);
		reflector.delegate(*t);
	}
}

/// Specialization of the non-intrusive reflect for boost::optional.
template<typename DerivedReflector, typename T>
void reflectWrite(ReflectorInterface<DerivedReflector>& r, boost::optional<T>& t)
{
	T v;
	DerivedReflector& reflector = static_cast<DerivedReflector&>(r);
	reflector.delegate(v);
	t.reset(v);
}

/** specialization for binary serialization, where we need to store at least
* a marker flag to indicate, whether the optional data is stored or not
*/
template<typename BinaryStream, typename T>
void reflectRead(BinarySerializer<BinaryStream>& r, boost::optional<T>& t)
{
	if (t) {
		uint8 flag = 1;
		r.member("Flag", flag, "");
		r.member("Content", *t, "");
	} else {
		uint8 flag = 0;
		r.member("Flag", flag, "");
	}
}

template<typename BinaryStream, typename T>
void reflectWrite(BinaryDeserializer<BinaryStream>& r, boost::optional<T>& t)
{
	uint8 flag = 0;
	r.member("Flag", flag, "");

	if(flag==1) {
		T v;
		r.member("Content", v, "");
		t.reset(v);
	}
	else
		t.reset();
}

template <typename DerivedReflector, typename T>
void reflect(ReflectorInterface<DerivedReflector>& iR, boost::optional<T>& t)
{
	DerivedReflector& reflector = static_cast<DerivedReflector&>(iR);
	splitReflect(reflector, t);
}

template<typename T, typename SerializerTag>
class IsTransparentSerializable<boost::optional<T>,SerializerTag> : public std::true_type {};

template<typename T>
class IsNotMetaSerializable<boost::optional<T>> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
