/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file array.hpp
 *    Serialization/Deserialization of boost::array.
 *
 *    The filename was chosen to match the boost standard.
 *    Instead of
 *      #include <boost/array.hpp>
 *    use
 *      #include <serialization/adapters/boost/array.hpp>
 *
 * @author Erik Einhorn
 * @date   12/02/2011
 */

#ifndef _MIRA_BOOST_ARRAY_HPP_
#define _MIRA_BOOST_ARRAY_HPP_

#ifndef Q_MOC_RUN
#include <boost/array.hpp>
#endif

#include <serialization/ReflectorInterface.h>
#include <serialization/Array.h>

#include <serialization/IsTransparentSerializable.h>
#include <serialization/IsCollection.h>

namespace mira {

////////////////////////////////////////////////////////////////////////////////

/// Specialization of the non-intrusive reflect for boost::array
template<typename Reflector, typename T, std::size_t N>
void reflectRead(Reflector& r, boost::array<T, N>& c)
{
	// store the size first
	uint32 count = c.size();
	serialization::ReflectCollectionCount<Reflector,boost::array<T, N>>::reflect(r, count);
	// store the elements as array
	serialization::PlainArray<T> array(c.data(), c.size());
	r.delegate(array);
}

/// Specialization of the non-intrusive reflect for boost::array
template<typename Reflector, typename T, std::size_t N>
void reflectWrite(Reflector& r, boost::array<T, N>& c)
{
	// restore the size first
	uint32 count;
	serialization::ReflectCollectionCount<Reflector,boost::array<T, N>>::reflect(r, count);
	// restore the elements as array
	serialization::PlainArray<T> array(c.data(), c.size());
	r.delegate(array);
}

/// non-intrusive reflect for boost::array
template<typename Reflector, typename T, std::size_t N>
void reflect(Reflector& r, boost::array<T, N>& c)
{
	splitReflect(r, c);
}

template<typename T, std::size_t N>
class IsCollection<boost::array<T, N>> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
