/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GetterSetter.h
 *    Provides definition for getters and setters that are used with
 *    the serialization framework.
 *
 * @author Erik Einhorn
 * @date   2010/07/06
 */

#ifndef _MIRA_GETTERSETTER_H_
#define _MIRA_GETTERSETTER_H_

#include <type_traits>

#ifndef Q_MOC_RUN
#include <boost/function.hpp>
#include <boost/bind.hpp>
#endif

#include <serialization/IsTransparentSerializable.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Holds a boost::function object to a special getter function
 * that must meet the signature "T method()".
 * Getter objects usually are created using the appropriate
 * getter() methods (see @ref GetterSetter.h)
 *
 * To call the stored getter method you can use the getter object
 * as a functor, e.g:
 *
 * \code
 * Getter<int> g = getter(myMethod); // create the getter
 *
 * int val = g(); // call the getter
 * \endcode
 *
 * Getters are used by the Serialization framework.
 *
 * @see @ref SerializationPage
 */
template<typename T>
class Getter
{
public:

	typedef T value_type;

public:

	Getter(boost::function<value_type ()> f) : fn(f) {}

	/// apply the getter, e.g. int val = mygetter.get();
	value_type get() const { return fn(); }

	/**
	 * cast operator to mimic cast behavior of underlying value type;
	 * applies the getter, e.g. int val = mygetter;
	 */
	operator value_type() const { return get(); }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		value_type tmp = fn();
		r.delegate(tmp);
	}

private:
	boost::function<value_type ()> fn;
};

///////////////////////////////////////////////////////////////////////////////

template <typename T, typename SerializerTag>
class IsTransparentSerializable<Getter<T>,SerializerTag> : public std::true_type {};

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for global or static class
 * methods returning the result by value.
 * e.g.:
 * \code
 *   int getValue() {}
 *   ...
 *
 *   getter(getValue)
 * \endcode
 */
template<typename T>
Getter<T> getter(T (*f)())
{
	return Getter<T>(f);
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for global or static class
 * methods returning the result by const reference.
 * e.g:
 * \code
 *   const int& getValue() {}
 *   ...
 *
 *   getter(getValue)
 * \endcode
 */
template<typename T>
Getter<T> getter(const T& (*f)())
{
	return Getter<T>(f);
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for non-const class methods returning
 * the result by value.
 * e.g:
 * \code
 *   class MyClass {
 *       int getValue() {}
 *       ...
 *
 *       getter(&MyClass::getValue, this)
 *   };
 * \endcode
 */
template<typename T, typename Class>
Getter<T> getter( T (Class::*f) (), Class* obj)
{
	return Getter<T>( boost::bind( f, boost::ref(*obj) ) );
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for non-const class methods returning
 * the value by const reference.
 * e.g.
 * \code
 *   class MyClass {
 *       const int& getValue() {}
 *       ...
 *
 *       getter(&MyClass::getValue, this)
 *   };
 * \endcode
 */
template<typename T, typename Class>
Getter<T> getter( const T& (Class::*f) (), Class* obj)
{
	return Getter<T>( boost::bind( f, boost::ref(*obj) ) );
}


/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for const class methods returning the
 * result by value.
 * e.g.
 * \code
 *   class MyClass {
 *       int getValue() const {}
 *       ...
 *
 *       getter(&MyClass::getValue, this)
 *   };
 * \endcode
 */
template<typename T, typename Class>
Getter<T> getter( T (Class::*f) () const, Class* obj)
{
	return Getter<T>( boost::bind( f, boost::ref(*obj) ) );
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for const class methods returning the
 * result by const reference.
 * e.g.
 * \code
 *   class MyClass {
 *       const int& getValue() const {}
 *       ...
 *
 *       getter(&MyClass::getValue, this)
 *   };
 * \endcode
 */
template<typename T, typename Class>
Getter<T> getter( const T& (Class::*f) () const, Class* obj)
{
	return Getter<T>( boost::bind( f, boost::ref(*obj) ) );
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for a lambda function.
 * e.g:
 * \code
 *     getter<size_t>([&](){ return myVector.size(); })
 * \endcode
 */
template<typename T, typename LambdaFn>
Getter<T> getter(LambdaFn fn)
{
	return Getter<T>(fn);
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for boost function or boost bind.
 *
 * e.g.
 * \code
 *   getter<int>(boost::bind(...))
 * \endcode
 */
template<typename T>
Getter<T> getter(boost::function<T ()> f)
{
	return Getter<T>(f);
}

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
template <typename T, typename TObject>
class GetterWithObject
{
public:

	GetterWithObject(boost::function<T (const TObject&)> f, const TObject& obj) :
		fn(f), object(obj) {}

public:

	T operator()() { return fn(object); }

private:

	boost::function<T (const TObject&)> fn;
	const TObject& object;
};

///@endcond


/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for a global function or static class
 * method that additionally is assigned with a certain object
 * of type TObject, i.e. when the Getter is called, the specified function
 * is called with the assigned object as parameter in order to return a
 * value of type T. This is often used for getters that convert
 * values or types.
 *
 * The passed function maps: TObject -> T
 *
 * e.g.
 *
 * \code
 *
 * float deg2rad(int v) { convert from integral degrees to floating point radian }
 *
 * class MyClass
 * {
 *     int myvalue;
 *
 *     getter<float>(deg2rad, myvalue)
 * };
 *
 * \endcode
 */
template<typename T, typename TObject>
Getter<T> getter(T (*f)(const TObject&), const TObject& object)
{
	return Getter<T>(GetterWithObject<T,TObject>(f,object));
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for a global function or static class
 * method that additionally is assigned with a certain object.
 * Same as above.
 */
template<typename T, typename TObject>
Getter<T> getter(T (*f)(TObject), const TObject& object)
{
	return Getter<T>(GetterWithObject<T,TObject>(f,object));
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for a boost bind function
 * that additionally is assigned with a certain object.
 * Same as above.
 */
template<typename T, typename TObject>
Getter<T> getter(boost::function<T(const TObject&)> f, const TObject& object)
{
	return Getter<T>(GetterWithObject<T,TObject>(f,object));
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Getter for a boost bind function
 * that additionally is assigned with a certain object.
 * Same as above.
 */
template<typename T, typename TObject>
Getter<T> getter(boost::function<T(TObject)> f, const TObject& object)
{
	return Getter<T>(GetterWithObject<T,TObject>(f,object));
}


/**
 * @ingroup SerializationModule
 *
 * Holds a boost::function object to a special setter function
 * that must meet the signature "void method(T)".
 * Setter objects usually are created using the appropriate
 * setter() methods (see @ref GetterSetter.h).
 *
 * To call the stored setter method you can use the setter object
 * as a functor, e.g:
 *
 * \code
 * Setter<int> s = setter(myMethod); // create the setter
 *
 * s(123); // call the setter
 * \endcode
 *
 * Setters are used by the Serialization framework.
 *
 * @see @ref SerializationPage
 */
template<typename T>
class Setter
{
public:
	typedef T value_type;

public:
	Setter(boost::function<void (const value_type&)> f) : fn(f) {}

	/// apply the setter, e.g. mysetter.set(1234);
	void set(const value_type& value) { fn(value); }

	/// mimic the assignment behavior of underlying type, e.g. mysetter = 1234;
	Setter& operator=(const value_type& value)
	{
		set(value);
		return *this;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		value_type tmp;
		r.delegate(tmp);
		set(tmp);
	}

private:

	boost::function<void (const value_type&)> fn;
};

template <typename T, typename SerializerTag>
class IsTransparentSerializable<Setter<T>,SerializerTag> : public std::true_type {};

/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for global or static class methods
 * taking the argument by const reference.
 * e.g.
 * \code
 *     void setValue(const int& value) {}
 *     ...
 *
 *     setter(setValue)
 * \endcode
 */
template<typename T>
Setter<T> setter(void (*f)(const T&))
{
	return Setter<T>(f);
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for global or static class methods
 * taking the argument by value.
 * e.g:
 * \code
 *     void setValue(int value) {}
 *     ...
 *
 *     setter(setValue)
 * \endcode
 */
template<typename T>
Setter<T> setter(void (*f)(T))
{
	return Setter<T>(f);
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for lambda function.
 * e.g:
 * \code
 *     setter([](int value){})
 * \endcode
 */
template<typename T, typename LambdaFn>
Setter<T> setter(LambdaFn fn)
{
	return Setter<T>(fn);
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for class methods taking the argument by const reference.
 * e.g.
 * \code
 *     class MyClass {
 *         void setValue(const int& value) {}
 *         ...
 *
 *         setter(&MyClass::setValue, this)
 *     }
 * \endcode */
template<typename T, typename Class>
Setter<T> setter( void (Class::*f) (const T&), Class* obj)
{
	return Setter<T>( boost::bind( f, boost::ref(*obj), _1 ) );
}

/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for class methods taking the argument by value.
 * e.g:
 * \code
 *     class MyClass {
 *         void setValue(int value) {}
 *         ...
 *
 *         setter(&MyClass::setValue, this)
 *     }

 * \endcode
 */
template<typename T, typename Class>
Setter<T> setter( void (Class::*f) (T), Class* obj)
{
	return Setter<T>( boost::bind( f, boost::ref(*obj), _1 ) );
}


/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for boost function or boost bind.
 *
 * e.g.
 * \code
 *   setter<int>(boost::bind(...))
 * \endcode
 */
template<typename T>
Setter<T> setter(boost::function<void (const T&)> f)
{
	return Setter<T>(f);
}

///@cond INTERNAL
template <typename T, typename TObject>
class SetterWithObject
{
public:

	SetterWithObject(boost::function<TObject (const T&)> f, TObject& obj) :
		fn(f), object(obj) {}

public:

	void operator()(const T& value) { object = fn(value); }

private:

	boost::function<TObject (const T&)> fn;
	TObject& object;
};
///@endcond

/**
 * @ingroup SerializationModule
 *
 * Creates a Setter for a global function or static class
 * method that additionally is assigned with a certain object
 * of type TObject, i.e. when the Setter is called, the specified function
 * is called with a value of type T and the result is assigned to the
 * assigned object.
 * This is often used for setters that convert values or types.
 *
 * The passed function maps: T -> TObject
 *
 * e.g.
 *
 * \code
 *
 * int rad2rad(float v) { convert from floating point radian to integral degrees}
 *
 * class MyClass
 * {
 *     int myvalue;
 *
 *     setter<float>(rad2deg, myvalue)
 * };
 *
 * \endcode
 */
template<typename T, typename TObject>
Setter<T> setter(TObject (*f)(const T&), TObject& object)
{
	return Setter<T>(SetterWithObject<T,TObject>(f,object));
}

/*
 * @ingroup SerializationModule
 *
 * Creates a Setter for a global function or static class
 * method that additionally is assigned with a certain object.
 * Same as above.
 */
template<typename T, typename TObject>
Setter<T> setter(TObject (*f)(T), TObject& object)
{
	return Setter<T>(SetterWithObject<T,TObject>(f,object));
}

/*
 * @ingroup SerializationModule
 *
 * Creates a Setter for a boost bind function
 * that additionally is assigned with a certain object.
 * Same as above.
 */
template<typename T, typename TObject>
Setter<T> setter(boost::function<TObject(const T&)> f, TObject& object)
{
	return Setter<T>(SetterWithObject<T,TObject>(f,object));
}

/*
 * @ingroup SerializationModule
 *
 * Creates a Setter for a boost bind function
 * that additionally is assigned with a certain object.
 * Same as above.
 */
template<typename T, typename TObject>
Setter<T> setter(boost::function<TObject(T)> f, TObject& object)
{
	return Setter<T>(SetterWithObject<T,TObject>(f,object));
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
