/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinarySerializer.h
 *    Binary serializer and deserializer.
 *
 * @author Erik Einhorn
 * @date   2010/07/03
 */

#ifndef _MIRA_BINARYSERIALIZER_H_
#define _MIRA_BINARYSERIALIZER_H_

#include <stream/BinaryStream.h>

#include <serialization/Serializer.h>
#include <serialization/Deserializer.h>

#include <serialization/BinarySerializerCodec.h>

#include <serialization/IsBitwiseSerializable.h>
#include <platform/Types.h>
#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Used by BinarySerializer and BinaryDeserializer
class BinarySerializerMixin
{
public:
	/**
	 * Pointer type that is stored as 1-byte marker before storing the pointer.
	 * @note For internal use within BinarySerializer and BinaryDeserializer only.
	 */
	enum PointerType
	{
		NULL_POINTER        = 0,
		POINTER_REFERENCE   = 1,
		NORMAL_POINTER      = 2,
		POLYMORPHIC_POINTER = 3
	};
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Serializer that uses BinaryOstream to serialize
 * the objects in binary format.
 *
 * It can be used for streaming data to tapes, for streaming data using TCP,
 * or for serializing the data into a binary buffer.
 * Depending on the used binary stream, properties like network byte
 * order can be specified via the binary stream object.
 *
 * Since this serializer operates on a BinaryOstream, it comes in two
 * flavors:
 * - \ref mira::BinaryStreamSerializer "BinaryStreamSerializer", which writes
 *   the data into stl conform streams.
 * - \ref mira::BinaryBufferSerializer "BinaryBufferSerializer", which stores
 *   the data in a Buffer.
 *
 * This serializer has a special serialize() method that supports storing
 * type information for better type safety.
 *
 * Example usage for BinaryBufferSerializer:
 * \code
 * #include <serialization/adapters/std/vector>
 * // the data to be serialized
 * std::vector<int> myValue;
 * ...
 *
 * // create a serializer that writes into a buffer
 * Buffer<uint8> buf;
 * BinaryBufferSerializer serializer(&buffer);
 *
 * // serialize the data
 * serializer.serialize(myValue);
 *
 * // buf now contains the serialized data.
 * \endcode
 *
 * Example usage for BinaryStreamSerializer:
 * \code
 * #include <serialization/adapters/std/vector>
 * // the data to be serialized
 * std::vector<int> myValue;
 * ...
 *
 * // create a binary serializer that writes into a file stream
 * ofstream ofs("myfile.bin");
 * BinaryStlOstream bos(ofs);
 * BinaryStreamSerializer serializer(bos);
 *
 * // serialize the data
 * serializer.serialize(myValue);
 * \endcode
 *
 * For the deserialization of binary content see
 * @ref mira::BinaryDeserializer "BinaryDeserializer".
 *
 * @see @ref SerializationPage, BinaryOstream, BinaryDeserializer
 */

struct BinarySerializerTag {};

template <typename Derived>
class BinarySerializer : public Serializer<Derived>
{
public:
	typedef BinarySerializerTag Tag;

public:

	int version(int version) {
		return this->This()->version(version);
	}

public:

	template <typename T>
	void write(const T* data, std::size_t count) {
		this->This()->write(data,count);
	}

public:

	/**
	 * Returns true, of there is a codec for the specified type T.
	 * In this case, the encode method can be used to encode the data using
	 * the codec.
	 */
	template <typename T>
	bool hasCodec() const {
#ifdef MIRA_LINUX
		return this->This()->template hasCodec<T>();
#else
		return this->This()->hasCodec<T>();
#endif
	}

	/**
	 * Encodes the specified object containing the data using a matching
	 * codec. The encoded data will be written directly into the binary output.
	 * If no codec was found, false is returned, and the caller must serialize
	 * the data manually without codec (in this case "NULL" is written as
	 * codec fourcc into the binary stream).
	 */
	template <typename T>
	bool codec(const T& obj)
	{
#ifdef MIRA_LINUX
		return this->This()->template codec(obj);
#else
		return this->This()->codec(obj);
#endif
	}
};


template <typename BinaryStream>
class ConcreteBinarySerializer : public BinarySerializer<ConcreteBinarySerializer<BinaryStream> >, public BinarySerializerMixin
{
	typedef BinarySerializer<ConcreteBinarySerializer<BinaryStream> > Base;

public:

	// We are not using human readable ids here. This will turn off the
	// whole id-handling system of the serialization/reflection framework
	// and dramatically improve the performance
	typedef boost::mpl::bool_<false> useHumanReadableIDs;

public:
	/**
	 * Create a new binary serializer based on the specified stream buffer
	 * object. The stream buffer object must be specified as
	 * pointer. If you use this class with BinaryStlOstream, then the
	 * stream buffer object must be of the type std::basic_streambuf<_CharT, _Traits>
	 * (see corresponding documentation of STL streams). If you use
	 * this class with BinaryBufferOstream, the stream buffer object
	 * must be of the type std::vector<char>.
	 *
	 * @note: The stream buffer object must exist as long as the
	 *		  serializer is used.
	 */
	ConcreteBinarySerializer(typename BinaryStream::streambuffer_pointer buffer) :
		mStream(buffer) {}

	/**
	 * Create a new binary serializer and assign it to the
	 * specified binary output stream.
	 *
	 * @note: The stream object must exist as long as the
	 *		  serializer is used.
	 */
	ConcreteBinarySerializer(BinaryStream& stream) : mStream(stream.rdbuf()) {}

public:

	/**
	 * Reassigns the specified stream buffer to this serializer. The
	 * stream buffer usually is specified in the constructor. This method
	 * can be used to change the assigned buffer.
	 */
	void reassign(typename BinaryStream::streambuffer_pointer buffer) {
		mStream = buffer;
	}

public:

	/**
	 * Provides a special serialize interface for the BinarySerializer.
	 * It allows to skip the name, since the name is not used by the binary
	 * serializer. Instead an optional second parameter is provided that allows
	 * to enable an automatic type check. If the type check is enabled the
	 * serializer will write the full typename into the stream before
	 * serializing the content. When deserializing the data, this type
	 * information is used to ensure that the data types match (type safety).
	 * If the type check is enabled here, it also must be enabled in the
	 * corresponding deserialize() call of the BinaryDeserializer and vice
	 * versa.
	 */
	template <typename T>
	void serialize(const T& value, bool enableTypeCheck = true)
	{
		if(enableTypeCheck) {
			std::string fulltypename = typeName<T>();
			Base::serialize("", fulltypename);
		}
		Base::serialize("", value);
	}

	int version(int version) {
		// serialize as atomic
		atomic(version);
		return version;
	}

	/** @name Visiting methods */
	//@{

	template<typename T>
	void atomic(T& member)
	{
		Base::atomic(member);
		// write out the value in binary format
		mStream << member;
	}

	//@}

	/*
	 * Pointers are stored with a prepended 1-byte marker that
	 * indicates the type of the pointer:
	 *    1 = PointerReference
	 * It is followed immediately by the object id (int) to the
	 * referenced object.
	 */
	void pointerReference(int referencedObjectID) {
		mStream << (uint8)POINTER_REFERENCE;
		mStream << referencedObjectID;
	}

	/*
	 * Pointers are stored with a prepended 1-byte marker that
	 * indicates the type of the pointer:
	 *    2 = NormalPointer
	 * It is followed by the serialized object.
	 */
	void pointerWithoutClassType() {
		mStream << (uint8)NORMAL_POINTER;
	}

	/*
	 * Pointers are stored with a prepended 1-byte marker that
	 * indicates the type of the pointer:
	 *    3 = PolymorphicPointer
	 * It is followed immediately by the string of the class
	 * type of the object and afterwards the serialized object
	 * follows.
	 */
	void pointerWithClassType(const std::string& type) {
		mStream << (uint8)POLYMORPHIC_POINTER;
		mStream << type;
	}

	/*
	 * Pointers are stored with a prepended 1-byte marker that
	 * indicates the type of the pointer:
	 *    0 = NullPointer
	 */
	void pointerNull() {
		mStream << (uint8)NULL_POINTER;
	}

	template<typename T>
	void invokeOverwrite(T& object)
	{
		Base::invokeOverwrite(object);
	}

	/**
	 * Specialized for Array to implement an optimized variant for
	 * serializing array, if possible (i.e. if object tracking is disabled and
	 * each value in the array can be serialized bitwise).
	 */
	template<typename T>
	void invokeOverwrite(serialization::PlainArray<T>& array)
	{
		if(this->template isTrackingEnabled<T>() || !IsBitwiseSerializable<T>::value) {
			// if tracking is enabled or the array's elements cannot be
			// serialized bitwise we cannot even think about using optimized
			// array serialization
			Base::invokeOverwrite(array);
			return;
		}

		// otherwise, USE the optimized variant and dump the whole array
		// into the stream as one block.
		const char* buffer = reinterpret_cast<const char*>(array.data);
		mStream.write(buffer, array.getSizeInBytes());
	}

public:

	template <typename T>
	void write(const T* data, std::size_t count) {
		mStream.write(reinterpret_cast<const char*>(data),count*sizeof(T));
	}

public:
	// Codec support


	/**
	 * Registers the specified codec instance in this binary serializer.
	 * Afterwards, the codec can be used by objects that are serialized and
	 * that support the codec.
	 * Note, that only one codec can be registered for each datatype, otherwise
	 * a XLogical exception will be thrown.
	 */
	void registerCodec(BinarySerializerCodecPtr codec) {
		if(!codec)
			return;

		TypeId type = codec->getSupportedTypeId();
		auto p = mCodecs.insert(std::make_pair(type, codec));
		if(!p.second)
			MIRA_THROW(XLogical, "A codec for the same type exists already");
	}

	/// Removes a previously registered codec
	void unregisterCodec(BinarySerializerCodecPtr codec) {
		if(!codec)
			return;

		TypeId type = codec->getSupportedTypeId();
		auto it = mCodecs.find(type);
		if(it==mCodecs.end())
			return; // not registered

		if(it->second!=codec)
			return; // not the same codec

		mCodecs.erase(it);
	}

	/**
	 * Returns true, of there is a codec for the specified type T.
	 * In this case, the encode method can be used to encode the data using
	 * the codec.
	 */
	template <typename T>
	bool hasCodec() const {
		return mCodecs.count(typeId<T>())!=0;
	}

	/**
	 * Encodes the specified object containing the data using a matching
	 * codec. The encoded data will be written directly into the binary output.
	 * If no codec was found, false is returned, and the caller must serialize
	 * the data manually without codec (in this case "NULL" is written as
	 * codec fourcc into the binary stream).
	 */
	template <typename T>
	bool codec(const T& obj)
	{
		BinarySerializerCodecPtr codec;
		BinarySerializerCodec::Fourcc s = BinarySerializerCodec::Fourcc::null();

		auto it = mCodecs.find(typeId<T>());
		if(it!=mCodecs.end()) {
			// found codec
			codec = it->second;
			s = codec->getFourcc();
		}

		// write the signature of the codec (or NULL if no codec was found)
		write(s.fourcc, sizeof(s.fourcc));

		if(!codec)
			return false;

		try {
			// encode the data using the codec
			const Buffer<uint8>& encodedData = codec->encode(obj);

			// write the encoded data immediately
			mStream << encodedData;

		} catch(...) {

			// was not able to encode data, so serialize it normally
			return false;
		}

		// done
		return true;
	}

private:
	BinaryStream mStream;

	std::map<TypeId, BinarySerializerCodecPtr> mCodecs;
};

/**
 * Typedef for BinarySerializer based on a Buffer. See BinarySerializer
 * for details.
 * @ingroup SerializationModule
 */
typedef ConcreteBinarySerializer<BinaryBufferOstream> BinaryBufferSerializer;

/**
 * Typedef for BinarySerializer based on STL streams. See BinarySerializer
 * for details.
 * @ingroup SerializationModule
 */
typedef ConcreteBinarySerializer<BinaryStlOstream>    BinaryStreamSerializer;

/**
 * @ingroup SerializationModule
 *
 * Deserializer that uses BinaryIstream to deserialize the objects from binary
 * format.
 * It can be used for streaming data from tapes, for streaming data using TCP,
 * or for deserializing the data from a binary buffer.
 * Depending on the used binary stream, properties like network byte
 * order can be specified via the binary stream object.
 *
 * Since this deserializer operates on a BinaryIstream, it comes in two
 * flavors:
 * - \ref mira::BinaryStreamDeserializer "BinaryStreamDeserializer", which reads
 *   the data from stl conform streams.
 * - \ref mira::BinaryBufferDeserializer "BinaryBufferDeserializer", which reads
 *   the data from a Buffer.
 *
 * This deserializer has a special deserialize() method that supports
 * type information for better type safety.
 *
 * Example usage for BinaryBufferDeserializer:
 * \code
 * Buffer<uint8> buffer;
 * ... // fill the buffer with data
 * BinaryBufferDeserializer deserializer(&buffer);
 *
 * // the object that will be filled with the content
 * std::vector<int> myValue;
 *
 * // deserialize the value
 * deserializer.deserialize(myValue);
 *
 * // myValue is now filled with the content that was stored
 * \endcode
 *
 * Example usage for BinaryStreamDeserializer:
 * \code
 * ifstream ifs("myfile.bin");
 * BinaryStlIstream bis(ifs);
 * BinaryStreamDeserializer deserializer(bis);
 *
 * // the object that will be filled with the content
 * std::vector<int> myValue;
 *
 * // deserialize the value
 * deserializer.deserialize(myValue);
 *
 * // myValue is now filled with the content that was stored
 * \endcode
 *
 * For the serialization of binary content see
 * @ref mira::BinarySerializer "BinarySerializer".
 *
 * @see @ref SerializationPage, BinaryIstream, BinarySerializer
 */
template <typename BinaryStream>
class BinaryDeserializer : public Deserializer<BinaryDeserializer<BinaryStream> >,
                           public BinarySerializerMixin
{
	typedef Deserializer<BinaryDeserializer<BinaryStream> > Base;

public:

	// We are not using human readable ids here. This will turn off the
	// whole id-handling system of the serialization/reflection framework
	// and dramatically improve the performance
	typedef boost::mpl::bool_<false> useHumanReadableIDs;

	/// A map of binary serialization codecs.
	typedef std::map<TypeId, BinarySerializerCodecPtr> CodecsMap;

public:

	/**
	 * Create a new binary deserializer based on the specified stream buffer
	 * object. The stream buffer object must be specified as
	 * pointer. If you use this class with BinaryStlIstream, then the
	 * stream buffer object must be of the type std::basic_streambuf<_CharT, _Traits>
	 * (see corresponding documentation of STL streams). If you use
	 * this class with BinaryBufferIstream, the stream buffer object
	 * must be of the type std::vector<char>.
	 *
	 * @note: The stream buffer object must exist as long as the
	 *		  deserializer is used.
	 */
	BinaryDeserializer(typename BinaryStream::streambuffer_pointer buffer) :
		mStream(buffer) {}


	/**
	 * Create a new binary deserializer and read the data from
	 * the specified binary stream.
	 *
	 * @note: The stream object must exist as long as the
	 *		  deserializer is used.
	 */
	BinaryDeserializer(BinaryStream& stream) : mStream(stream.rdbuf()) {}

public:

	/**
	 * Reassigns the specified stream buffer to this deserializer. The
	 * stream buffer usually is specified in the constructor. This method
	 * can be used to change the assigned buffer.
	 */
	void reassign(typename BinaryStream::streambuffer_pointer buffer) {
		mStream = buffer;
	}

public:

	/**
	 * Provides a special deserialize interface for the BinaryDeserializer.
	 * It allows to skip the name, since the name is not used by the binary
	 * serializer. Instead an optional second parameter is provided that allows
	 * to enable an automatic type check. If the type check is enabled the
	 * deserializer will read the full typename from the stream before
	 * serializing the content. If the types do not match a XBadCast exception
	 * is thrown.
	 * If the type check is enabled here, it also must be enabled in the
	 * corresponding serialize() call of the BinarySerializer and vice
	 * versa.
	 */
	template <typename T>
	void deserialize(T& value, bool enableTypeCheck = true)
	{
		if(enableTypeCheck) {
			std::string fulltypename;
			try {
				Base::deserialize("", fulltypename);
			}
			catch (...) {
				MIRA_THROW(XBadCast, "Failed reading type from binary data while " <<
				    "deserializing into a variable of type '" << typeName<T>() << "'");
			}

			if(fulltypename != typeName<T>())
				MIRA_THROW(XBadCast, "Cannot deserialize the type '" << fulltypename <<
				          "' into a variable of type '" << typeName<T>() << "'");
		}
		Base::deserialize("", value);
	}

public:

	int version(int expectedVersion) {
		// deserialize as atomic
		int version;
		atomic(version);
		if (version > expectedVersion)
			MIRA_THROW(XIO, "Trying to deserialize binary data of a newer version (" << version <<
				") of a type into an older version (" << expectedVersion << ").");
		return version;
	}

	template<typename T>
	void atomic(T& member)
	{
		Base::atomic(member);
		// read in the value from binary stream
		mStream >> member;
		if(mStream.fail())
			MIRA_THROW(XIO, "Failed to read member of type '" << typeName<T>()
			          << "' from binary stream");
	}

	/**
	 * Overwrites Deserializer::pointer
	 *
	 * Reads the 1-byte marker to identify the type of the pointer:
	 *  NULL_POINTER        =0,
	 *  POINTER_REFERENCE   =1,
	 *  NORMAL_POINTER      =2,
	 *  POLYMORPHIC_POINTER =3
	 *
	 * If it is null pointer the pointer is set to NULL, if it is
	 * a reference then it reads the id string that was
	 * stored by the Serializer in pointerReference().
	 * If it is a PolymorphicPointer it reads the class type
	 * and calls the pointer() method of the Deserializer base class
	 * to deserialize the full object.
	 * Otherwise if it is a normal pointer it calls the pointer() method
	 * of the Deserializer base class to deserialize the full object.
	 */
	template<typename T>
	void pointer(T* &pointer)
	{
		// we always store a marker before each pointer
		// in Serializer::pointerReference / pointerNoReference
		uint8 pointerType8U;
		mStream >> pointerType8U;

		PointerType pointerType = (PointerType) pointerType8U;

		switch(pointerType)
		{
			case NULL_POINTER:
				pointer=NULL;
				break;

			case POINTER_REFERENCE:
			{
				int ref;
				mStream >> ref; // read the ref id
				// ... and resolve it
				pointer = this->template resolveReference<T>(ref);
				break;
			}

			case NORMAL_POINTER:
				// we have a "normal" pointer, so deserialize it
				mClassType.clear(); // we have no class type information here
				Base::pointer(pointer);
				break;

			case POLYMORPHIC_POINTER:
				// we have a "polymorphic" pointer, so get class type
				mStream >> mClassType;
				// and deserialize it
				Base::pointer(pointer);
				break;
		}
	}

	std::string pointerClassType() {
		return mClassType;
	}


	template<typename T>
	void invokeOverwrite(T& object)
	{
		Base::invokeOverwrite(object);
	}

	/**
	 * Specialized for Array to implement an optimized variant for
	 * deserializing an array, if possible (i.e. if object tracking is enabled and
	 * each value in the array can be serialized bitwise).
	 */
	template<typename T>
	void invokeOverwrite(serialization::PlainArray<T>& array)
	{
		if(this->template isTrackingEnabled<T>() || !IsBitwiseSerializable<T>::value) {
			// if tracking is enabled or the array's elements cannot be
			// serialized bitwise we cannot even think about using optimized
			// array deserialization
			Base::invokeOverwrite(array);
			return;
		}

		// otherwise, USE the optimized variant and read the whole array
		// from the stream as one block.
		char* buffer = reinterpret_cast<char*>(array.data);
		mStream.read(buffer, array.getSizeInBytes());
	}

public:

	template <typename T>
	void read(T* data, std::size_t count) {
		mStream.read(reinterpret_cast<char*>(data),count*sizeof(T));
	}

public:
	// codec support

	/**
	 * Supported for compatibility with BinarySerializer in a common reflect
	 * method. Always returns false for deserializers.
	 */
	template <typename T>
	bool hasCodec() const {
		//return mCodecs.count(typeId<T>())!=0;
		return false;
	}

	/**
	 * Decodes the specified object from the serialized data.
	 * If no codec was used to serialize the data, false is returned, and the
	 * caller must deserialize the data manually without codec.
	 * If no codec for decoding the data exists, an exception is thrown.
	 */
	template <typename T>
	bool codec(T& obj)
	{
		BinarySerializerCodecPtr codec;
		BinarySerializerCodec::Fourcc s;

		// read the codec fourcc
		read(s.fourcc, sizeof(s.fourcc));

		if(s==BinarySerializerCodec::Fourcc::null())
			return false; // no codec was used by serializer

		// try to find appropriate codec within our created codecs
		auto it = mCodecs.find(typeId<T>());
		if(it==mCodecs.end() || it->second->getFourcc()!=s) {
			// codec not found, so we must create one
			codec = BinarySerializerCodec::createCodec<T>(s);
			assert(codec);
			mCodecs[typeId<T>()] = codec;
		} else
			codec = it->second; // use existing codec

		assert(codec->getFourcc()==s);
		assert(codec->getSupportedTypeId()==typeId<T>());

		// read the encoded data
		Buffer<uint8> encodedData;
		mStream >> encodedData;

		// and decode
		codec->decode(encodedData, obj);

		// done
		return true;
	}

	/// Return the map of currently known codecs.
	const CodecsMap& getCodecs() const {
		return mCodecs;
	}

	/// Set the map of known codecs.
	void setCodecs(const CodecsMap& codecs) {
		mCodecs = codecs;
	}

private:

	BinaryStream mStream;
	std::string  mClassType;

	CodecsMap mCodecs;
};

/**
 * Typedef for BinaryDeserializer based on a Buffer. See BinaryDeserializer
 * for details.
 * @ingroup SerializationModule
 */
typedef BinaryDeserializer<BinaryBufferIstream> BinaryBufferDeserializer;

/**
 * Typedef for BinaryDeserializer based on a stl stream. See BinaryDeserializer
 * for details.
 * @ingroup SerializationModule
 */
typedef BinaryDeserializer<BinaryStlIstream>    BinaryStreamDeserializer;

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
