/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCInvoker.h
 *    Implements the RPCInvoker and TRPCInvoker classes.
 *
 * @author Erik Einhorn
 * @date   2010/11/07
 */

#ifndef _MIRA_RPCINVOKER_H__
#define _MIRA_RPCINVOKER_H__

#include <type_traits>
#ifndef Q_MOC_RUN
#include <boost/preprocessor/repetition.hpp>
#include <boost/function.hpp>
#include <boost/bind.hpp>
#endif

#include <error/Exception.h>

#include <rpc/RPCError.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 *
 * Base of all TRPCInvoker classes which are templates to support different
 * RPC backends.
 */
class RPCInvoker
{
public:
	virtual ~RPCInvoker() {}
};

/**
 * @ingroup RPCModule
 *
 * Invoker that is used to invoke an RPC method call for a special backend.
 * The concrete implementations of the invokers are generated by boost::preprocessor
 * and the macros at the end of this file.
 * For each RPC method that is reflected in the service object an RPCInvoker
 * object is created. The function is as follows:
 * -# Upon construction of the RPCInvoker object the method pointer and the
 *    pointer to the object (for non-static methods) are stored in the constructor
 * -# When calling invoke they RPCInvokers extracts the parameters from the
 *    specified request, uses boost::bind to build an unary function from the
 *    method stored in step 1, and invokes the function. Finally, the result
 *    of the call is stored in the response.
 *
 * @see @ref ServiceRPCPage
 */
template <typename Backend>
class TRPCInvoker : public RPCInvoker
{
public:
	/**
	 * object is pointer to service object whose method is called
	 */
	virtual void invoke(typename Backend::ServerRequest& request,
			            typename Backend::ServerResponse& response) = 0;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * The helper is required since we have methods with and without
 * return value. For non-void return types it serializes the
 * return value. Additionally, there is a specialized version for a return type
 * of void which just calls the method without taking care of the return
 * value (which is just void).
 */
template <typename R>
struct RPCInvokeHelper
{
	template <typename Backend>
	static void invoke(typename Backend::ServerResponse& response, boost::function<R()> fn)
	{
		typedef typename std::remove_reference<R>::type plainR;
		plainR* res = NULL;
		try {
			res = new plainR(fn()); // invoke the function and store the result
		}
		catch (Exception& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex.what(),
			                         ex.callStack(), getCurrentThreadID());
			return;
		}
		catch (std::exception& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex.what());
			return;
		}
		catch (...) {
			response.returnException(RPC_EXCEPTION_IN_CALL, "Unknown exception");
			return;
		}

		response.template returnResult<R>(*res);
		delete res; // delete temporary result
	}
};

//////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for return type "void"
 */
template <>
struct RPCInvokeHelper<void>
{
	template <typename Backend>
	static void invoke(typename Backend::ServerResponse& response, boost::function<void()> fn)
	{
		try {
			fn(); // invoke the function ( we have no result here)
		}
		catch (Exception& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex.what(),
			                         ex.callStack(), getCurrentThreadID());
			return;
		}
		catch (std::exception& ex) {
			response.returnException(RPC_EXCEPTION_IN_CALL, ex.what());
			return;
		}
		catch (...) {
			response.returnException(RPC_EXCEPTION_IN_CALL, "Unknown exception");
			return;
		}
		response.returnVoid();
	}
};

//////////////////////////////////////////////////////////////////////////////

/*
The following code uses boost preprocessor for generating the Invokers
For different number of parameters using preprocessor meta programming.

The following code is generated for different numbers of P0 ... PN:

template<typename Backend, typename R, typename P0, typename P1>
struct RPCInvoker2 : public TRPCInvoker<Backend> {
	RPCInvoker2(boost::function<R (P0 , P1)> f) : fn(f) {}
	virtual void invoke(Backend::ServerRequest& request,
		Backend::ServerResponse& response)
	{
		P0  p0; request.getParameter(p0);
		P1  p1; request.getParameter(p1);
		RPCInvokeHelper<R>::invoke(response, boost::bind(fn,p0,p1));
	}
	boost::function<R ( P0 , P1)> fn;
};

template<typename Backend, typename R, typename P0, typename P1>
struct RPCInvoker2_memfn : public TRPCInvoker<Backend> {
	RPCInvoker2_memfn(boost::function<R (Class*,P0,P1)> f, Class* obj) :
		fn(f), This(obj) {}
	virtual void invoke(Backend::ServerRequest& request,
		Backend::ServerResponse& response)
	{
		P0  p0; request.getParameter(p0);
		P1  p1; request.getParameter(p1);
		RPCInvokeHelper<R>::invoke(response, boost::bind(fn,This,p0,p1));
	}
	boost::function<R (Class* , P0 , P1)> fn;
	Class* This;
};
*/
#define RPCGEN_GET_PARAMETER_(z, n, _)                                         \
	typename std::remove_const<typename std::remove_reference<P##n>::type>::type p##n; \
	request.getParameter(p##n);

#define RPCGEN_GET_PARAMETER(z, n, _)                                          \
	BOOST_PP_REPEAT_ ## z(n, RPCGEN_GET_PARAMETER_, nil)

#define RPCGEN_INVOKER(z,n,_)                                                  \
template<typename Backend, typename R                                          \
			BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                   \
struct RPCInvoker##n : public TRPCInvoker<Backend> {                           \
	RPCInvoker##n(boost::function<R (                                          \
								BOOST_PP_ENUM_PARAMS_Z(z,n,P))> f) : fn(f) {}  \
	virtual void invoke(typename Backend::ServerRequest& request,              \
                        typename Backend::ServerResponse& response) {          \
		RPCGEN_GET_PARAMETER(z,n,nil)                                          \
		RPCInvokeHelper<R>::template invoke<Backend>(response, boost::bind(fn  \
									BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,p)));  \
	}                                                                          \
	boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> fn;                     \
};

#define RPCGEN_INVOKER_MEMFN(z,n,_)                                            \
template<typename Backend, typename R, typename Class                          \
			BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                   \
struct RPCInvoker##n##_memfn : public TRPCInvoker<Backend> {                   \
	RPCInvoker##n##_memfn(boost::function<R (Class*                            \
						BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P))> f,            \
                        Class* obj) : fn(f) , This(obj) {}                     \
	virtual void invoke(typename Backend::ServerRequest& request,              \
                        typename Backend::ServerResponse& response) {          \
		RPCGEN_GET_PARAMETER(z,n,nil)                                          \
		RPCInvokeHelper<R>::template invoke<Backend>(response, boost::bind(fn, This     \
									BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,p)));  \
	}                                                                          \
	boost::function<R (Class* BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P))> fn;     \
	Class* This;                                                               \
};

BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_INVOKER,nil)
BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_INVOKER_MEMFN,nil)

#undef RPCGEN_GET_PARAMETER_
#undef RPCGEN_GET_PARAMETER
#undef RPCGEN_INVOKER
#undef RPCGEN_INVOKER_MEMFN

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_RPCINVOKER_H_ */
