/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LinearInterpolator.h
 *    Filter that can be used for linear interpolation.
 *
 * @author Erik Einhorn
 * @date   2010/10/03
 */

#ifndef _MIRA_LINEARINTERPOLATOR_H_
#define _MIRA_LINEARINTERPOLATOR_H_

#include <assert.h>

#include <limits>

#include <math/IsApprox.h>
#include <math/Lerp.h>
#include <math/IntervalFilter.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * 1D linear interpolator.
 * This interpolator internally uses the lerp() method, and hence can be
 * applied for all types that are supported by the lerp() method or implement
 * their own lerp() method.
 *
 * @ingroup MathModule
 */
class LinearInterpolator : public IntervalFilter
{
public:
	bool canExtrapolate() const { return true; }
	int samples() const         { return 2; }
	int samplesBefore() const   { return 1; }
	int samplesAfter() const    { return 1; }

	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	Ty apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx)
	{
		assert(x.size()==y.size());
		assert(x.size()==2);

		double dx = (double)x.back() - (double)x.front();
		// check if dx is close to 0, then return the first
		if(isApprox(dx,0.0, std::numeric_limits<double>::epsilon()))
			return y.front();

		return lerp(y.front(),y.back(), ((double)xx-(double)x.front())/dx);
	}
};

//////////////////////////////////////////////////////////////////////////////

/**
 * 1D linear interpolator.
 * This interpolator internally uses the lerp() method, and hence can be
 * applied for all types that are supported by the lerp() method or implement
 * their own lerp() method. Does not support extrapolation.
 *
 * @ingroup MathModule
 */
class LinearInterpolatorNoExtrapolation : public IntervalFilter
{
public:
	bool canExtrapolate() const { return false; }
	int samples() const         { return 2; }
	int samplesBefore() const   { return 1; }
	int samplesAfter() const    { return 1; }

	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	Ty apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx)
	{
		assert(x.size()==y.size());
		assert(x.size()==2);

		double dx = (double)x.back() - (double)x.front();
		// check if dx is close to 0, then return the first
		if(isApprox(dx,0.0, std::numeric_limits<double>::epsilon()))
			return y.front();

		return lerp(y.front(),y.back(), ((double)xx-(double)x.front())/dx);
	}
};

//////////////////////////////////////////////////////////////////////////////

/**
 * 1D linear interpolator.
 * This interpolator internally uses the lerp() method, and hence can be
 * applied for all types that are supported by the lerp() method or implement
 * their own lerp() method. Does support extrapolation only by taking the nearest neighbour -
 * meaning the first or last sample of the supplied interval if alpha < 0 or alpha > 1.
 *
 * @ingroup MathModule
 */
class LinearInterpolatorNearestNeighbourExtrapolator : public IntervalFilter
{
public:

	bool canExtrapolate() const { return true; }
	int samples() const         { return 2; }
	int samplesBefore() const   { return 1; }
	int samplesAfter() const    { return 1; }

	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	Ty apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx)
	{
		assert(x.size()==y.size());
		assert(x.size()==2);
		
		// handle extrapolation case
		if (xx < x.front())
			return y.front();
		if (xx > x.back())
			return y.back();

		double dx = (double)x.back() - (double)x.front();
		// check if dx is close to 0, then return the first
		if(isApprox(dx,0.0, std::numeric_limits<double>::epsilon()))
			return y.front();		
	
		return lerp(y.front(),y.back(),((double)xx-(double)x.front())/dx);
	}
};

//////////////////////////////////////////////////////////////////////////////

/**
 * 1D linear interpolator.
 * This interpolator internally uses the lerp() method, and hence can be
 * applied for all types that are supported by the lerp() method or implement
 * their own lerp() method. Does support extrapolation with supplied limit.
 * If alpha for lerp is < limit or > 1 + limit an exception is thrown.
 * A limit of 0 means no extrapolation allowed.
 * A limit of 1 means extrapolation about the same length of the sample interval 
 * is allowed, and so on...
 *
 * @ingroup MathModule
 */
class LinearInterpolatorExtrapolationLimit : public IntervalFilter
{
public:
	LinearInterpolatorExtrapolationLimit(double limit = 1.0) :
		mLimit(limit) {}

	bool canExtrapolate() const { return true; }
	int samples() const         { return 2; }
	int samplesBefore() const   { return 1; }
	int samplesAfter() const    { return 1; }

	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	Ty apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx)
	{
		assert(x.size()==y.size());
		assert(x.size()==2);

		double dx = (double)x.back() - (double)x.front();
		// check if dx is close to 0, then return the first
		if(isApprox(dx,0.0, std::numeric_limits<double>::epsilon()))
			return y.front();

		// check extrapolation case
		double alpha = ((double)xx-(double)x.front())/dx;
		if (alpha < -mLimit || alpha > (1 + mLimit))
			MIRA_THROW(XRuntime,"LinearInterpolatorExtrapolationLimit: Alpha (" << alpha << ") for lerp out of specified limits (" << -mLimit << "," << 1 + mLimit << ")");
	
		return lerp(y.front(),y.back(), alpha);
	}
	
private:
	double mLimit;
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_LINEARINTERPOLATOR_H_ */
