/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgPixel.h
 *    Classes to access pixels of images with variable type and channels.
 *
 * @author Erik Einhorn
 * @date   2010/10/21
 */

#ifndef _MIRA_IMGPIXEL_H_
#define _MIRA_IMGPIXEL_H_

#include <opencv/cxcore.h>

#include <image/Color.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief base interface for ImgPixel classes
 *	This class wraps typed cv::Vect to represent an ImagePixel
 *	The Vect is typed by template parameter T and channels are set by template
 *	parameter Channels.
 */
template<typename T, int Channels>
class ImgPixelBase {
public:
	/// access to index i of vector
	T operator[](int i) const {
		return v[i];
	}
	/// const access to index i of vector
	T& operator[](int i) {
		return v[i];
	}

	/// access to index i of vector
	T operator()(int i) const {
		return v[i];
	}
	/// const access to index i of vector
	T& operator()(int i) {
		return v[i];
	}

public:
	bool operator==(const ImgPixelBase& other) const {
		return v == other.v;
	}

public:
	///casts internal cv::Vect to cv::Scalar
	operator cv::Scalar() const {
		return (cv::Scalar) v;
	}

protected:
	cv::Vec<T, Channels> v;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief dynamic ImgPixel class
 * template ImgPixel class providing flexible types and channels.
 * Channels of this class are >= 5, since cases 1-4 are specialized in extra
 * classes for efficiency and convenience reasons.
 * @ingroup ImageModule
 */
template<typename T, int Channels>
class ImgPixel: public ImgPixelBase<T, Channels> {
public:
	/// The default constructor.
	ImgPixel() {
	}

	///setting cv::Vector from Scalar
	explicit ImgPixel(const cv::Scalar& s) {
		this->v = s;
	}
};

/**
 * @brief specialization of PixelBase class for 1 channel
 */
template<typename T>
class ImgPixel<T, 1> : public ImgPixelBase<T, 1> {
public:
	/// The default constructor.
	ImgPixel() {
	}

	///constructor setting the value of the pixel
	ImgPixel(T v0) {
		this->v[0] = v0;
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief specialization of PixelBase class for 2 channels
 */
template<typename T>
class ImgPixel<T, 2> : public ImgPixelBase<T, 2> {
public:
	/// The default constructor.
	ImgPixel() {
	}

	///constructor setting the values of the pixel
	ImgPixel(T v0, T v1) {
		this->v[0] = v0;
		this->v[1] = v1;
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief specialization of PixelBase class for 3 channels
 */
template<typename T>
class ImgPixel<T, 3> : public ImgPixelBase<T, 3> {
public:
	/// The default constructor.
	ImgPixel() {
	}

	///constructor setting the values of the pixel
	ImgPixel(T v0, T v1, T v2) {
		this->v[0] = v0;
		this->v[1] = v1;
		this->v[2] = v2;
	}

	/// construction from a color of ColorBase
	ImgPixel(const ColorBase& color) {
		Color::RGB c = color.toRGB();
		this->v[0] = (T) (c.b * 255.0f);
		this->v[1] = (T) (c.g * 255.0f);
		this->v[2] = (T) (c.r * 255.0f);
	}
};

/**
 * @brief specialization of PixelBase class for 4 channels
 */
template<typename T>
class ImgPixel<T, 4> : public ImgPixelBase<T, 4> {
public:
	ImgPixel() {
	}
	///constructor setting the values of the pixel
	ImgPixel(T v0, T v1, T v2, T v3) {
		this->v[0] = v0;
		this->v[1] = v1;
		this->v[2] = v2;
		this->v[3] = v3;
	}

	/** constructor that takes ColorBase, internally the cv::Scalar conversion
	 *  operator of the color is used.
	 */
	ImgPixel(const ColorBase& color) {
		cv::Scalar c = color;
		this->v = c;
	}

};

///////////////////////////////////////////////////////////////////////////////

}// namespace mira

/// specialize cv::DataType for our ImgPixel and inherit from cv::DataType<Vec>

namespace mira { namespace detail {

// make sure depth and type are defined in cv::DataType<ImgPixel>
// (deprecated  from base classes in OpenCV 3.3):
// hierarchy below checks for existence of cv::DataType<Vec>::depth,
// cv::DataType<Vec>::type separately and adds them if required,
// with original definitions taken from OpenCV

MIRA_MEMBER_DETECTOR(depth)

template<typename T, int Channels,
         bool HasDepth = MIRA_HAS_MEMBER_TEMPLATE(T, depth)::value>
class CVDataTypeVecEnsureDepth :
		public cv::DataType<cv::Vec<T, Channels> > {
};

template<typename T, int Channels>
class CVDataTypeVecEnsureDepth<T, Channels, false> :
		public cv::DataType<cv::Vec<T, Channels> > {

	typedef cv::DataType<cv::Vec<T, Channels> > Base;
public:
	enum { depth = cv::DataType<typename Base::channel_type>::depth };
};

MIRA_MEMBER_DETECTOR(type)

template<typename T, int Channels,
         bool HasType = MIRA_HAS_MEMBER_TEMPLATE(T, type)::value>
class CVDataTypeVecEnsureDepthAndType :
		public CVDataTypeVecEnsureDepth<T, Channels> {
};

template<typename T, int Channels>
class CVDataTypeVecEnsureDepthAndType<T, Channels, false> :
		public CVDataTypeVecEnsureDepth<T, Channels> {

	typedef CVDataTypeVecEnsureDepth<T, Channels> Base;
public:
	enum { type = CV_MAKETYPE(Base::depth, Base::channels) };
};

}}

namespace cv {

template<typename T, int Channels>
class DataType<mira::ImgPixel<T, Channels> > : 
	public mira::detail::CVDataTypeVecEnsureDepthAndType<T, Channels> {
};

}

///////////////////////////////////////////////////////////////////////////////

#endif
