/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file StripedStorage.h
 *    Stores data sequentially into different files (striped) providing a
 *    reliable way to retrieve data later from one of the files
 *
 * @author Tim Langner
 * @date   2012/11/02
 */

#ifndef _MIRA_STRIPEDSTORAGE_H_
#define _MIRA_STRIPEDSTORAGE_H_

#include <error/Exceptions.h>
#include <serialization/XMLSerializer.h>
#include <serialization/BinarySerializer.h>
#include <utils/Path.h>
#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class StripedStorageBase
{
public:
	StripedStorageBase() : filePath(""), fileNameBase("data"), extension(""), maxFiles(2), mCurrentFileIdx(0) {}

	/// Create a striped storage providing a base file name and the maximum number of files
	StripedStorageBase(std::string baseName, std::string extension, uint32 maxFiles) :
		filePath(""), fileNameBase(baseName), extension(extension), maxFiles(maxFiles), mCurrentFileIdx(0) {}

	/// Create a striped storage providing a path, a base file name and the maximum number of files
	StripedStorageBase(const Path& directory, std::string baseName, std::string extension, uint32 maxFiles) :
		filePath(directory), fileNameBase(baseName), extension(extension), maxFiles(maxFiles), mCurrentFileIdx(0) {}

public:
	/// Returns the filename (including path) of the file with the specified nr.
	Path generateFilename(uint32 fileNr) const
	{
		std::string filename = fileNameBase + toString(fileNr) + extension;
		return filePath / filename;
	}

	/// Returns a list of filenames (including paths) for all files that are used for persistent storage.
	std::list<Path> getFiles() const
	{
		std::list<Path> list;
		for (uint32 i = 0; i < maxFiles; ++i)
			list.push_back(generateFilename(i));
		return list;
	}

	/// Delete all files that were stored by the persistent storage.
	void deleteAllFiles()
	{
		auto files = getFiles();
		foreach(auto f, files)
			boost::filesystem::remove(f);
		mCurrentFileIdx = 0;
	}

	Path filePath;             ///< The path where the files are stored
	std::string fileNameBase;  ///< The base filename
	std::string extension;	   ///< The file name extension
	uint32 maxFiles;           ///< Maximum number of used files

protected:
	uint32 mCurrentFileIdx;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Stores data sequentially into different files (striped) providing a reliable
 * way to retrieve data later from one of the files even if some of the files
 * are corrupted (e.g. caused by program termination during write operation).
 */
class StripedStorageXML : public StripedStorageBase
{
public:
	StripedStorageXML() : StripedStorageBase() { extension = ".xml"; }
	StripedStorageXML(std::string baseName, uint32 maxFiles) : StripedStorageBase(baseName, ".xml", maxFiles) {}
	StripedStorageXML(const Path& directory, std::string baseName, uint32 maxFiles) :
		StripedStorageBase(directory, baseName, ".xml", maxFiles) {}

public:
	/**
	 * Stores the given data in one of the files adding a timestamp and returns
	 * the filename of the written file.
	 */
	template <typename T>
	Path store(const T& data, const Time& at = Time::now())
	{
		// Store the content in an XML file.
		const Path filename = generateFilename(mCurrentFileIdx);

		XMLDom xml;
		XMLSerializer s(xml);
		s.serialize(fileNameBase, data);
		s.serialize("Date", at);
		xml.saveToFile(filename);

		MIRA_LOG(TRACE) << "StripedStorageXML: Wrote to file " << filename << " with timestamp " << at.toUnixTimestamp();

		// Switch to the next file at the end, after everything was okay.
		mCurrentFileIdx = (mCurrentFileIdx + 1) % maxFiles;

		return filename;
	}

	/**
	 * Retrieves the data from one of the files.
	 * If more than one file contain valid data the data with the newest
	 * timestamp is returned.
	 * @param validCheckFunc optional function for checking if restored data is valid
	 * @return A pair of data and timestamp
	 */
	template <typename T>
	std::pair<T, Time> retrieve(bool (*validCheckFunc)(const T&) = NULL) const
	{
		std::pair<T, Time> newestData;
		std::pair<T, Time> pendingData;

		newestData.second = Time::unixEpoch();
		bool found = false;

		for(uint32 i = 0; i < maxFiles; ++i)
		{
			const Path filename = generateFilename(i);

			try
			{
				MIRA_LOG(TRACE) << "StripedStorageXML: Trying to restore from file " << filename;

				XMLDom xml;
				xml.loadFromFile(filename);

				XMLDeserializer ds(xml);
				ds.deserialize("Date", pendingData.second);

				if(pendingData.second <= newestData.second)
					MIRA_THROW(XLogical, "Data timestamp below required freshness data, aborting");

				// We have found data that is newer that the one we have restored so far
				ds.deserialize(fileNameBase.c_str(), pendingData.first);
				if(!validCheckFunc || validCheckFunc(pendingData.first))
				{
					newestData = pendingData;
					found = true;
				}
			} catch(const Exception& ex) {
				MIRA_LOG(TRACE) << "StripedStorageXML: Could not restore from file " << filename << ", exception: " << ex.what();
				// ignore and continue with next file
			}
		}

		// We have no file that contains valid data
		if(!found)
			MIRA_THROW(XIO, "Unable to retrieve persistent information");

		return newestData;
	}
};

///////////////////////////////////////////////////////////////////////////////

class StripedStorageBinary : public StripedStorageBase
{
public:
	StripedStorageBinary() : StripedStorageBase() { extension = ".bin"; }
	StripedStorageBinary(std::string baseName, uint32 maxFiles) : StripedStorageBase(baseName, ".bin", maxFiles) {}
	StripedStorageBinary(const Path& directory, std::string baseName, uint32 maxFiles) :
		StripedStorageBase(directory, baseName, ".bin", maxFiles) {}

public:
	/**
	 * Stores the given data in one of the files adding a timestamp and returns
	 * the filename of the written file.
	 */
	template <typename T>
	Path store(const T& data, const Time& at = Time::now())
	{
		// Store the content in an XML file.
		const Path filename = generateFilename(mCurrentFileIdx);

		std::ofstream ofs(filename.c_str());
		BinaryStlOstream bos(ofs);
		BinaryStreamSerializer s(bos);
		s.serialize(at, false);
		s.serialize(data, false);
		s.serialize(at, false); // again for sanity check in retrieve

		MIRA_LOG(TRACE) << "StripedStorageBinary: Wrote to file " << filename << " with timestamp " << at.toUnixTimestamp();

		// Switch to the next file at the end, after everything was okay.
		mCurrentFileIdx = (mCurrentFileIdx + 1) % maxFiles;

		return filename;
	}

	/**
	 * Retrieves the data from one of the files.
	 * If more than one file contain valid data the data with the newest
	 * timestamp is returned.
	 * @param validCheckFunc optional function for checking if restored data is valid
	 * @return A pair of data and timestamp
	 */
	template <typename T>
	std::pair<T, Time> retrieve(bool (*validCheckFunc)(const T&) = NULL) const
	{
		std::pair<T, Time> newestData;
		std::pair<T, Time> pendingData;

		newestData.second = Time::unixEpoch();
		bool found = false;

		for(uint32 i = 0; i < maxFiles; ++i)
		{
			const Path filename = generateFilename(i);

			try
			{
				MIRA_LOG(TRACE) << "StripedStorageBinary: Trying to restore from file " << filename;

				std::ifstream ifs(filename.c_str());
				BinaryStlIstream bis(ifs);
				BinaryStreamDeserializer ds(bis);
				ds.deserialize(pendingData.second, false);

				if(pendingData.second <= newestData.second)
					MIRA_THROW(XLogical, "Data timestamp below required freshness data, aborting");

				// We have found data that is newer that the one we have restored so far
				ds.deserialize(pendingData.first, false);
				Time sanityTime;
				ds.deserialize(sanityTime, false);
				if(sanityTime != pendingData.second)
					MIRA_THROW(XLogical, "Deserialized time sanity check failed");

				if(!validCheckFunc || validCheckFunc(pendingData.first))
				{
					newestData = pendingData;
					found = true;
				}
			} catch(const Exception& ex) {
				MIRA_LOG(TRACE) << "StripedStorageBinary: Could not restore from file " << filename << ", exception: " << ex.what();
				// ignore and continue with next file
			}
		}

		// We have no file that contains valid data
		if(!found)
			MIRA_THROW(XIO, "Unable to retrieve persistent information");

		return newestData;
	}
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
