/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CommonTapeRenderer.C
 *    Common renderer plugins.
 *
 * @author Tim Langner, Christof Schroeter
 * @date   2011/12/30
 */

#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

#include <image/Img.h>
#include <widgets/QtUtils.h>
#include <visualization/ColormapProperty.h>

#include <TapeTextRenderer.h>
#include <TapeRendererImage.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

VideoCodecDialog::VideoCodecDialog(QWidget* parent) :
	QDialog(parent)
{
	setWindowTitle("Select codec");
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
	QHBoxLayout* hLayout = new QHBoxLayout();
	mCodec = new QComboBox(this);
	// query all codec classes from the class factory:
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap codecClasses = video::AbstractVideoOutput::CLASS().getDerivedClasses();
	foreach(ClassMap::value_type i, codecClasses)
	{
		if(i.second.isAbstract())
			continue;
		if (i.second.getMetaInfo("Codec").empty())
			continue;
		mCodec->addItem(i.first.c_str());
	}
	if (mCodec->count() == 0)
		MIRA_THROW(XInvalidParameter, "No codec for video output found.");

	hLayout->addWidget(new QLabel("Codec", this));
	hLayout->addWidget(mCodec);
	layout->addLayout(hLayout);

	connect(mCodec, SIGNAL(currentIndexChanged(const QString &)), this, SLOT(changeCodec(const QString &)));

	mPropertyEditor = new PropertyEditor(this);
	mPropertyEditor->setHideSingleRootNode(true);

	changeCodec(mCodec->currentText());

	layout->addWidget(mPropertyEditor);

	mButtonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);

	connect(mButtonBox,  SIGNAL(accepted()), this, SLOT(accept()));
	connect(mButtonBox,  SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(mButtonBox);
}

void VideoCodecDialog::changeCodec(const QString & codec)
{
	auto videoClass = ClassFactory::instance().getClassByIdentifier(codec.toStdString());
	mVideoOut.reset(dynamic_cast<video::AbstractVideoOutput*>(videoClass.newInstance()));
	if(!mVideoOut)
		MIRA_THROW(XInvalidParameter, "Could not create codec '" << codec.toStdString() << "'");

	mPropertyEditor->clear();
	PropertySerializer s;
	PropertyNode* p = s.reflectProperties("Codec", mVideoOut);
	mPropertyEditor->addProperty(p);
}

///////////////////////////////////////////////////////////////////////////////

template <typename ImageType>
class TapeImageRenderer : public TapeDataRenderer
{
public:

	TapeImageRenderer()
	{
		mDataHeight = 45;
		mAspectRatio = 4.0f/3.0f;
		mIndicatorHeight = 5;
		mImageMin = 0.0f;
		mImageMax = -1.0f;
		mFeatures = RENDER | INFO;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		TapeDataRenderer::reflect(r);
		r.property("Colormap", mColormap, "The color palette", ColormapProperty("mira::GrayscaleColormap"));
		r.property("Min", mImageMin, "min. value in the image (only valid for float images)", 0.0);
		r.property("Max", mImageMax, "max. value in the image (only valid for float and uint16 images)", -1.0);
	}

	virtual int getDataWidth() const {
		return (int)(mAspectRatio*(float)mDataHeight);
	}

	virtual void fillContextMenu(QMenu* menu, TapeChannelInfo& info,
	                             TapeChannelInfo::DataMap::iterator& message,
	                             int64 time);
	virtual void executeAction(QAction* action, TapeChannelInfo& info,
	                           TapeChannelInfo::DataMap::iterator& message,
	                           int64 time);

	virtual Typename getDatatype() { return typeName<ImageType>(); }

	virtual QSize renderMessage(QPainter* painter, const QRect& maxRect,
	                            TapeChannelInfo& info,
	                            TapeChannelInfo::DataMap::iterator& message);

	QString getMessageInfo(TapeChannelInfo& info,
	                       TapeChannelInfo::DataMap::iterator& message);

protected:

	ColormapProperty mColormap;
	float mImageMin;
	float mImageMax;

	float mAspectRatio;
};

//////////////////////////////////////////////////////////////////////////////

template <typename ImageType>
void TapeImageRenderer<ImageType>::fillContextMenu(QMenu* menu, TapeChannelInfo& info,
                                        TapeChannelInfo::DataMap::iterator& message,
                                        int64 time)
{
	TapeDataRenderer::fillContextMenu(menu, info, message, time);
	menu->addAction("Export image")->setEnabled(message != info.data.end());
	menu->addAction("Export all images");
	int64 selected = mEditor->getSelectionEnd() - mEditor->getSelectionStart();
	menu->addAction("Export selected images")->setEnabled(selected > 0);
	menu->addAction("Export all images as video");
	menu->addAction("Export selected images as video")->setEnabled(selected > 0);
}


template <typename ImageType>
void exportImageToFile(TapeChannelInfo::DataMap::iterator& message, const Path& dir)
{
	ImageType data;
	BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&message->second.getData()));
	bs.deserialize(data, false);
	std::string fileName = MakeString() << message->first << ".jpg";
	cv::imwrite((dir / fileName).string(), data);
}

template <typename ImageType>
void TapeImageRenderer<ImageType>::executeAction(QAction* action, TapeChannelInfo& info,
                                      TapeChannelInfo::DataMap::iterator& message,
                                      int64 time)
{
	Path directory;
	if (action->text() == "Export image" ||
		action->text() == "Export all images" ||
		action->text() == "Export selected images")
	{
		QString dirName = QFileDialog::getExistingDirectory(mEditor, "Directory");
		if (dirName.isEmpty())
			return;
		directory = dirName.toStdString();
	}
	if (action->text() == "Export image")
	{
		exportImageToFile<ImageType>(message, directory);
	}
	else if (action->text() == "Export all images")
	{
		for(auto it = info.data.begin(); it != info.data.end(); ++it)
			exportImageToFile<ImageType>(it, directory);
	}
	else if (action->text() == "Export selected images")
	{
		auto iStart = info.data.lower_bound(mEditor->getSelectionStart());
		auto iEnd = info.data.upper_bound(mEditor->getSelectionEnd());
		for ( ; iStart != iEnd; ++iStart)
			exportImageToFile<ImageType>(iStart, directory);
	}
	else if (action->text() == "Export all images as video" ||
			 action->text() == "Export selected images as video")
	{
		VideoCodecDialog codecDialog(mEditor);
		if (codecDialog.exec() == 0)
			return;

		QString fileName = QFileDialog::getSaveFileName(mEditor, "Select video file");
		if (fileName.isEmpty())
			return;

		video::AbstractVideoOutputPtr videoOut = codecDialog.getVideoOut();
		videoOut->open(fileName.toStdString());

		auto iStart = info.data.begin();
		auto iEnd = info.data.end();
		if (action->text() == "Export selected images as video")
		{
			iStart = info.data.lower_bound(mEditor->getSelectionStart());
			iEnd = info.data.upper_bound(mEditor->getSelectionEnd());
		}
		Time recTime = mEditor->getRecordingTime().toLocal();
		for ( ; iStart != iEnd; ++iStart)
		{
			ImageType data;
			BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&iStart->second.getData()));
			bs.deserialize(data, false);
			Time timestamp = recTime + Duration::microseconds(iStart->first);
			videoOut->encode(data, timestamp);
		}
		videoOut->close();
	}
	else
		TapeDataRenderer::executeAction(action, info, message, time);
}

template <typename ImageType>
QSize TapeImageRenderer<ImageType>::renderMessage(QPainter* painter, const QRect& maxRect,
                                       TapeChannelInfo& info,
                                       TapeChannelInfo::DataMap::iterator& message)
{
	ImageType data;
	BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&message->second.getData()));
	bs.deserialize(data, false);
	ImageType img;
	mAspectRatio = (float)data.width() / (float)data.height();
	QRect imageRect(maxRect.x()+1, maxRect.y(), getDataWidth()-1, maxRect.height());
	cv::resize(data, img, cv::Size2i(imageRect.width(), imageRect.height()));
	QImage qimg = QtUtils::toQImage(img, mImageMin, mImageMax);

	if(qimg.format()==QImage::Format_Indexed8)
	{
		if(mColormap.isValid())
			qimg.setColorTable(mColormap.getColorTable());
	}
	painter->drawImage(imageRect, qimg);
	painter->setPen(QPen(info.color));
	painter->drawLine(maxRect.x(), maxRect.top(), maxRect.x(), maxRect.bottom());
	return QSize(getDataWidth(), maxRect.height());
}


template <typename ImageType>
QString TapeImageRenderer<ImageType>::getMessageInfo(TapeChannelInfo& info,
                                          TapeChannelInfo::DataMap::iterator& message)
{
	ImageType data;
	BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&message->second.getData()));
	bs.deserialize(data, false);
	return QString("%1x%2x%3").arg(data.width()).arg(data.height()).arg(data.channels());
}

///////////////////////////////////////////////////////////////////////////////

class UntypedImageRenderer : public TapeImageRenderer<Img<>>
{
	MIRA_META_OBJECT(UntypedImageRenderer,
	                ("Name", "Image")
	                ("Description", "Renders data as image"))
};

class Image8U1Renderer : public TapeImageRenderer<Img8U1>
{
	MIRA_META_OBJECT(Image8U1Renderer,
	                ("Name", "Image")
	                ("Description", "Renders data as image"))
};

class Image8U3Renderer : public TapeImageRenderer<Img8U3>
{
	MIRA_META_OBJECT(Image8U3Renderer,
	                ("Name", "Image")
	                ("Description", "Renders data as image"))
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

//////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::UntypedImageRenderer, mira::TapeDataRenderer)
MIRA_CLASS_SERIALIZATION(mira::Image8U1Renderer, mira::TapeDataRenderer)
MIRA_CLASS_SERIALIZATION(mira::Image8U3Renderer, mira::TapeDataRenderer)

