/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeTextRenderer.h
 *    .
 *
 * @author Tim Langner
 * @date   2012/01/05
 */

#ifndef _MIRA_TAPETEXTRENDERER_H_
#define _MIRA_TAPETEXTRENDERER_H_

#include <TypedTapeDataRenderer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

inline QSize renderText(QPainter* painter, const QRect& maxRect,
                 TapeChannelInfo& info, const QString& message)
{
	const QFontMetrics& fm = painter->fontMetrics();
	int strLen = fm.width(message)+10;
	if (strLen > maxRect.width())
		return QSize(0, maxRect.height());
	QRect textRect(maxRect.x(), maxRect.y(), strLen, maxRect.height());
	painter->drawLine(maxRect.x(), maxRect.top(), maxRect.x(), maxRect.bottom());
	painter->setPen(QPen(info.color.darker()));
	painter->drawText(textRect, Qt::AlignCenter | Qt::AlignCenter, message);
	return QSize(strLen, maxRect.height());
}

//////////////////////////////////////////////////////////////////////////////

template <typename T>
class TapeTextRenderer : public TypedTapeDataRenderer<T>
{
public:

	TapeTextRenderer()
	{
		this->mIndicatorHeight = 5;
		this->mDataHeight = 20;
		this->mFeatures |= TapeDataRenderer::RENDER;
	}

	virtual int getDataWidth() const { return 20; }

	virtual QSize renderMessage(QPainter* painter, const QRect& maxRect,
	                            TapeChannelInfo& info,
	                            TapeChannelInfo::DataMap::iterator& message);
};

template<typename T>
QSize TapeTextRenderer<T>::renderMessage(QPainter* painter, const QRect& maxRect,
                                         TapeChannelInfo& info,
                                         TapeChannelInfo::DataMap::iterator& message)
{
	try
	{
		T data;
		BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&message->second.getData()));
		bs.deserialize(data, false);
		JSONSerializer js;
		json::Value json = js.serialize(data);
		if (!this->mMember.empty())
			json = json::getElement(json, this->mMember);
		QString s(json::write(json, true).c_str());
		return renderText(painter, maxRect, info, s);
	}
	catch(Exception& ex)
	{
		return renderText(painter, maxRect, info, ex.message().c_str());
	}
	return QSize(0, maxRect.height());
}

#define MIRA_REGISTER_TEXTRENDERER(name, type)                \
class name : public TapeTextRenderer<type> {                 \
	MIRA_META_OBJECT(name,                                    \
	                ("Name", "Text")                         \
	                ("Description", "Renders data as text")) \
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
