/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AddVersionTest.C
 *    Test cases for miratape addversion.
 *
 * @author Christof Schroeter
 * @date   2018/05/15
 */

#include <boost/test/unit_test.hpp>

#include <utils/Stamped.h>
#include <math/UniformRandomGenerator.h>
#include <platform/Process.h>

#include <serialization/MetaSerializer.h>

#include <fw/Tape.h>

#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/pair>

#define TAPENAME "test.tape"
#define TAPENAME_ADDEDVERSION "test_addedversion.tape"

namespace mira { namespace tapetest {

////////////////////////////////////////////////////////////////////////////////

bool gEnableVersion;

class ClassA
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		if (gEnableVersion) {    // we want to write this without version and
			                     // later check reading it with version
			int v = r.version(3);
			BOOST_CHECK(v == 2); // because we will set the version to 2 in
			                     // the tape after it is written
		}

		r.member("Int", mInt, "");
		r.member("Float", mFloat, "");
		r.member("Int8", mInt8, "");
		r.member("Bool", mBool, "");
	}

	bool operator==(const ClassA& other) const
	{
		return (mInt == other.mInt) &&
		       (mFloat == other.mFloat) &&
		       (mInt8 == other.mInt8) &&
		       (mBool == other.mBool);
	}

public:

	int mInt;
	float mFloat;
	uint8 mInt8;
	bool mBool;
};

std::ostream& operator<<(std::ostream& stream, const ClassA& o)
{
	return stream << "[ " << o.mInt << ", " << o.mFloat << ", "
	                      << o.mInt8 << ", " << (o.mBool?"true":"false") << " ]";
}
////////////////////////////////////////////////////////////////////////////////

void checkTape(const Tape& tape, std::vector<std::string> channelnames, int messages)
{
	Tape::ChannelMap channels = tape.getChannels();
	BOOST_CHECK_EQUAL(channels.size(), channelnames.size());
	foreach(auto c, channelnames)
		BOOST_CHECK_EQUAL(channels[c].messages.size(), messages);
}

template <typename T>
void writeObject(Tape* tape, const T& object,
                 const std::string& channel, MetaTypeDatabase& metaDB)
{
	static MetaSerializer metaSerializer(metaDB);

	Stamped<T> o(object, Time::now(), "");
	TypeMetaPtr meta = metaSerializer.addMeta(object);
	tape->write(channel, o, false, meta, metaDB);
}

template <typename T>
void readMessage(Tape* tape, const Tape::MessageIndex& index, Stamped<T>& oValue)
{
	Buffer<uint8> buffer;
	Duration time;
	tape->readMessage(index, oValue.frameID, oValue.sequenceID, buffer, time);
	oValue.timestamp = tape->getStartTime() + time;

	BinaryBufferDeserializer d(&buffer);
	d.deserialize(oValue.internalValueRep(), false);
}

template <typename T>
void readMessage(Tape* tape, const std::string& channel, int index, Stamped<T>& oValue)
{
	Tape::ChannelMap channels = tape->getChannels();
	readMessage(tape, channels[channel].messages[index], oValue);
}

////////////////////////////////////////////////////////////////////////////////

/// add version to the class that is the channel type
BOOST_AUTO_TEST_CASE(SimpleAddVersionTest)
{
	UniformRandomGenerator<int> rndI(1, 255);
	UniformRandomGenerator<float> rndF(1.0f, 255.0f);

	MetaTypeDatabase metaDB;

	std::vector<ClassA> objectsA;

	for (int n = 0; n < 10; ++n) {
		ClassA objectA;
		objectA.mInt = rndI();
		objectA.mFloat = rndF();
		objectA.mInt8 = rndI();
		objectA.mBool = rndI() % 2;

		objectsA.push_back(objectA);
	}

	gEnableVersion = false;

	{
		Tape t;
		t.open(TAPENAME, Tape::WRITE);

		for (int n = 0; n < 10; ++n)
			writeObject(&t, objectsA[n], "ChannelA", metaDB);

		t.close();
	}
	{
		Stamped<ClassA> o;

		Tape t;
		t.open(TAPENAME, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelA"}), 10);

		for (int n = 0; n < 10; ++n) {
			readMessage(&t, "ChannelA", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsA[n]);
		}
	}

	gEnableVersion = true;

	{
		Stamped<ClassA> o;

		Tape t;
		t.open(TAPENAME, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelA"}), 10);

		try {
			for (int n = 0; n < 10; ++n) {
				readMessage(&t, "ChannelA", n, o);

				// it's ok to fail reading with an XIO
				// if, however, reading succeeds, the result must not be what was written!
				BOOST_CHECK(!(o.value() == objectsA[n]));
			}
		}
		catch(XIO& ex) {}
	}

	// add the version number to all occurrences of ClassA in the tape
	executeProcess("miratape copy " TAPENAME " " TAPENAME_ADDEDVERSION
	               " --addversion mira::tapetest::ClassA=2");

	{
		Stamped<ClassA> o;

		Tape t;
		t.open(TAPENAME_ADDEDVERSION, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelA"}), 10);

		for (int n = 0; n < 10; ++n) {
			readMessage(&t, "ChannelA", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsA[n]);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

class ClassB : public ClassA
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Int16", mInt16, "");
		ClassA::reflect(r);
	}

	bool operator==(const ClassB& other) const
	{
		return (mInt16 == other.mInt16) && ClassA::operator==(other);
	}

public:

	int16 mInt16;
};

std::ostream& operator<<(std::ostream& stream, const ClassB& o)
{
	return stream << "[ " << (ClassA)o << ", " << o.mInt16 << " ]";
}

class ClassC : public ClassA
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		ClassA::reflect(r);
		r.member("Int64", mInt64, "");
	}

	bool operator==(const ClassC& other) const
	{
		return (mInt64 == other.mInt64) && ClassA::operator==(other);
	}

public:

	int64 mInt64;
};

std::ostream& operator<<(std::ostream& stream, const ClassC& o)
{
	return stream << "[ " << (ClassA)o << ", " << o.mInt64 << " ]";
}

class ClassD
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Pair", mPair, "");
	}

	bool operator==(const ClassD& other) const
	{
		return (mPair == other.mPair);
	}

public:
	std::pair<ClassB, ClassC> mPair;
};

std::ostream& operator<<(std::ostream& stream, const ClassD& o)
{
	return stream << print(o.mPair);
}

class ClassE
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Vector", mVector, "");
	}

	bool operator==(const ClassE& other) const
	{
		return (mVector == other.mVector);
	}

public:
	std::vector<ClassD> mVector;
};

std::ostream& operator<<(std::ostream& stream, const ClassE& o)
{
	return stream << print(o.mVector);
}

////////////////////////////////////////////////////////////////////////////////

// add version to a class that is a part of an arbitrarily complex structured channel type
BOOST_AUTO_TEST_CASE(ComplexAddVersionTest)
{
	UniformRandomGenerator<int> rndI(1, 255);
	UniformRandomGenerator<float> rndF(1.0f, 255.0f);

	MetaTypeDatabase metaDB;

	std::vector<ClassE> objectsE;

	for (int n = 0; n < 10; ++n) {
		ClassE objectE;

		int l = rndI();
		for (int i = 0; i < l; ++i) {
			ClassD objectD;
			objectD.mPair.first.mInt = rndI();
			objectD.mPair.first.mFloat = rndF();
			objectD.mPair.first.mInt8 = rndI();
			objectD.mPair.first.mBool = rndI() % 2;
			objectD.mPair.first.mInt16 = rndI();
			objectD.mPair.second.mInt = rndI();
			objectD.mPair.second.mFloat = rndF();
			objectD.mPair.second.mInt8 = rndI();
			objectD.mPair.second.mBool = rndI() % 2;
			objectD.mPair.second.mInt64 = rndI();

			objectE.mVector.push_back(objectD);
		}

		objectsE.push_back(objectE);
	}

	gEnableVersion = false;

	{
		Tape t;
		t.open(TAPENAME, Tape::WRITE);

		for (int n = 0; n < 5; ++n)
			writeObject(&t, objectsE[n], "ChannelE1", metaDB);
		for (int n = 5; n < 10; ++n)
			writeObject(&t, objectsE[n], "ChannelE2", metaDB);

		t.close();
	}
	{
		Stamped<ClassE> o;

		Tape t;
		t.open(TAPENAME, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelE1", "ChannelE2"}), 5);

		for (int n = 0; n < 5; ++n) {
			readMessage(&t, "ChannelE1", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n]);
		}
		for (int n = 5; n < 10; ++n) {
			readMessage(&t, "ChannelE2", n-5, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n]);
		}
	}

	gEnableVersion = true;

	{
		Stamped<ClassE> o;

		Tape t;
		t.open(TAPENAME, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelE1", "ChannelE2"}), 5);

		try {
			for (int n = 0; n < 5; ++n) {
				readMessage(&t, "ChannelE1", n, o);

				// it's ok to fail reading with an XIO
				// if, however, reading succeeds, the result must not be what was written!
				BOOST_CHECK(!(o.value() == objectsE[n]));
			}
			for (int n = 5; n < 10; ++n) {
				readMessage(&t, "ChannelE2", n-5, o);
				BOOST_CHECK(!(o.value() == objectsE[n]));
			}

		}
		catch(XIO& ex) {}
	}

	// add the version number to all occurrences of ClassB and ClassC in the tape
	// (both inherit the version member from ClassA)
	executeProcess("miratape copy " TAPENAME " " TAPENAME_ADDEDVERSION ".tmp"
	               " --addversion mira::tapetest::ClassB=2,1");
	executeProcess("miratape copy " TAPENAME_ADDEDVERSION ".tmp " TAPENAME_ADDEDVERSION
	               " --addversion mira::tapetest::ClassC=2");

	{
		Stamped<ClassE> o;

		Tape t;
		t.open(TAPENAME_ADDEDVERSION, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelE1", "ChannelE2"}), 5);

		for (int n = 0; n < 5; ++n) {
			readMessage(&t, "ChannelE1", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n]);
		}
		for (int n = 5; n < 10; ++n) {
			readMessage(&t, "ChannelE2", n-5, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n]);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

}}
