/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageGroup.C
 *    Implementation of PackageGroup.h
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */
#include "core/PackageGroup.h"
#include "core/Package.h"
#include "core/Dependency.h"

#include <serialization/Serialization.h>

#include <boost/algorithm/string/case_conv.hpp>

#include <fstream>

using namespace std;

MIRA_CLASS_SERIALIZATION(mira::PackageGroup,mira::Object);

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PackageGroup::PackageGroup() : mType(UNSPECIFIED), mParentPackage( NULL )
{
}

PackageGroup::PackageGroup( string const& name ) :
	mName( name ), mType(UNSPECIFIED), mParentPackage( NULL )
{
}

PackageGroup::~PackageGroup() {
	clearChildren();
}

/**
 * @brief Delete all child packages and clear child list.
 */
void PackageGroup::clearChildren() {
	foreach( PackageGroup* sub, mSubPackages) {
		delete sub;
	}
	mSubPackages.clear();
}

void PackageGroup::removeChild( PackageGroup* child )
{
	vector<PackageGroup*>::iterator it = mSubPackages.begin();
	for ( ; it != mSubPackages.end(); it++ ) {
		if ( (*it) == child ) {
			mSubPackages.erase( it );
			return;
		}
	}
}

void PackageGroup::add( PackageGroup* other )
{
	if ( other ) {
		// maybe a group with that name already exists?
		PackageGroup* group = findGroup( other->mName, true );
		if ( group ) {
			group->add( other );
			return;
		}
		// try to merge package with identical package
		vector<Package*> packages = findPackages( other->mName );
		foreach( Package* package, packages ) {
			if ( package->isIdentical( other ) ) {
				package->merge( other );
				return;
			}
		}
		// if a single package with the same name already
		// exists, we should build a new group 
		if ( packages.size() == 1 ) {
			Package* package = packages[0];
			assert( !findGroup( other->mName, true ) );
			PackageGroup* newGroup = new PackageGroup( package->mName );
			package->mParentPackage->exchangePackage( package, newGroup );
			assert( findGroup( other->mName, true ) );

			newGroup->addChild( package );
			newGroup->addChild( other );
			return;
		}
		
		addChild( other );
	}
}

void PackageGroup::addChild( PackageGroup* other )
{
	mSubPackages.push_back( other );
	other->mParentPackage = this;
}

void PackageGroup::merge( PackageGroup* other )
{
	if ( other ) {
		assert( other->getClass().getIdentifier()
				== PackageGroup::CLASS().getIdentifier() );
		mSubPackages.insert( mSubPackages.end(),
				other->mSubPackages.begin(), other->mSubPackages.end() );
		delete other;
	}
}

vector<Package*> PackageGroup::findPackages( string const& name, bool recursive ) const
{
	vector<Package*> tReturn;
	foreach( PackageGroup* group, mSubPackages ) {
		Package* package = dynamic_cast<Package*>( group );
		if ( package && group->mName == name )
			tReturn.push_back( package );
		if ( recursive ) {
			vector<Package*> tSubReturn
				= group->findPackages( name, true );
			tReturn.insert(tReturn.end(), tSubReturn.begin(), tSubReturn.end() );
		}
	}
	return tReturn;
}

PackageGroup* PackageGroup::getIdentical( PackageGroup const* other,
                                          Type ignoredFlags ) const
{
	foreach( PackageGroup* group, mSubPackages ) {
		if ( other->isIdentical( group, ignoredFlags ) ) {
			return group;
		}
		PackageGroup* tReturn = group->getIdentical( other, ignoredFlags );
		if ( tReturn )
			return tReturn;
	}
	return NULL;
}

void PackageGroup::getAllIdentical( PackageGroup const* other,
                                    vector<PackageGroup*>& identical,
                                    Type ignoredFlags ) const
{
	foreach( PackageGroup* group, mSubPackages ) {
		if ( other->isIdentical( group, ignoredFlags ) ) {
			identical.push_back( group );
		}
		group->getAllIdentical( other, identical, ignoredFlags );
	}
}

PackageGroup* PackageGroup::findGroup( std::string const& name, bool recursive ) const
{
	foreach( PackageGroup* group, mSubPackages ) {
		if ( group->getClass().getIdentifier() == PackageGroup::CLASS().getIdentifier() && group->mName == name )
			return group;
		if ( recursive ) {
			PackageGroup* tSubReturn
				= group->findGroup( name, true );
			if ( tSubReturn )
				return tSubReturn;
		}
	}
	return NULL;
}

PackageGroup* PackageGroup::createCopy() const
{
	PackageGroup* tReturn = getClass().newInstance();
	tReturn->mName = mName;
	tReturn->mType = mType;
	tReturn->mDescription = mDescription;
	tReturn->mParentPackage = mParentPackage;
	foreach( PackageGroup const* sub, mSubPackages ) {
		PackageGroup* tNewSub = sub->createCopy();
		tReturn->addChild( tNewSub );
	}
	return tReturn;
}

PackageGroup* PackageGroup::createDependency() const
{
	PackageGroup* tReturn = NULL;
	if ( getClass().getIdentifier() == Package::CLASS().getIdentifier() )
		tReturn = Dependency::CLASS().newInstance();
	else
		tReturn = getClass().newInstance();
	tReturn->mName = mName;
	tReturn->mType = mType;
	tReturn->mDescription = mDescription;
	tReturn->mParentPackage = mParentPackage;
	foreach( PackageGroup const* sub, mSubPackages ) {
		PackageGroup* tNewSub = sub->createDependency();
		tReturn->addChild( tNewSub );
	}
	return tReturn;
}

bool PackageGroup::operator<(const PackageGroup& other) const
{
	return mName < other.mName;
}

bool PackageGroup::isPrimary() const
{
	return false;
}

std::string PackageGroup::getShortDescription() const
{
	if ( mDescription.find(". ") != std::string::npos ) {
		return mDescription.substr(0, mDescription.find(". ")+1);
	}
	return mDescription;
}

bool PackageGroup::isIdentical( PackageGroup const* other, Type ignoredFlags ) const
{
	return (mName == other->mName)
			&& (mType | ignoredFlags) == (other->mType | ignoredFlags );
}

bool PackageGroup::canBeGroupedWith( PackageGroup const* other) const
{
	return mName == other->mName;
}

std::vector<Package*> PackageGroup::getPrimarySubPackages() const
{
	std::vector<Package*> primaryPackages;
	foreach( PackageGroup* package, mSubPackages ) {
		if ( package->getClass().getIdentifier()
				== Package::CLASS().getIdentifier() ) {
			Package* source = dynamic_cast<Package*>( package );
			assert( source );
			if ( source->isPrimary() )
				primaryPackages.push_back( source );
		}
	}
	return primaryPackages;
}

void PackageGroup::exchangePackage( PackageGroup* oldAddress, PackageGroup* newAddress )
{
	foreach( PackageGroup*& sub, mSubPackages ) {
		if ( sub == oldAddress ) {
			sub = newAddress;
			newAddress->mParentPackage = this;
			return;
		}
	}
}

void PackageGroup::selectStdRepo( Database* db )
{
	foreach( PackageGroup* sub, mSubPackages ) {
		sub->selectStdRepo( db );
	}
}

string PackageGroup::getTypeString() const
{
	switch ( mType ) {
		case UNSPECIFIED : return "UNSPECIFIED";break;
		case ETC : return "ETC";break;
		case BINARY : return "BINARY";break;
		case SOURCE : return "SOURCE";break;
		default: return "";
	}
}

///////////////////////////////////////////////////////////////////////////////

}

