/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TrajectoryVisualizaion.C
 *    Visualization of trajectories.
 *
 * @author Erik Einhorn
 * @date   2014/07/16
 */

#include <widgets/OgreUtils.h>

#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreManualObject.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/LineStripObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/SetterNotify.h>
#include <serialization/DefaultInitializer.h>

#include <transform/Pose.h>

#include <robot/Trajectory.h>

#include <image/Colormap.h>
#include <visualization/ColormapProperty.h>

using namespace mira::robot;

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename TTrajectory>
class TrajectoryVisualizationBase :  public Visualization3DBasic<TTrajectory>
{
	typedef TTrajectory Trajectory;
	typedef Visualization3DBasic<TTrajectory> Base;
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:

	TrajectoryVisualizationBase() :
		Base("Trajectory"),
		mLineObject(NULL)
	{
		MIRA_INITIALIZE_THIS;
	}

	virtual ~TrajectoryVisualizationBase()
	{
		if (Visualization3D::getSite() == NULL)
			return;
		delete mLineObject;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);
		r.property("Line Width", mLineWidth, setterNotify(mLineWidth, &TrajectoryVisualizationBase::update, this), "The width of the line",
		           0.1f);
		r.property("MinVelocity", mMinVelocity, setterNotify(mMinVelocity, &TrajectoryVisualizationBase::update, this), "Velocity which gets minimum color of palette", 0.0, PropertyHints::minimum(0.0) );
		r.property("MaxVelocity", mMaxVelocity, setterNotify(mMaxVelocity, &TrajectoryVisualizationBase::update, this), "Velocity which gets maximum color of palette", 0.0, PropertyHints::minimum(0.0) );
		r.property("Colormap", mColorMap, setter(&TrajectoryVisualizationBase::setColormap, this), "The color palette",
		           ColormapProperty("mira::RedBlueColormap"));
		r.property("StartIndex", mStartIndex, setterNotify(mStartIndex, &TrajectoryVisualizationBase::update, this), "indices before this point are ignored, if >= 0",
		           -1);
		r.property("EndIndex", mEndIndex, setterNotify(mEndIndex, &TrajectoryVisualizationBase::update, this), "indices after this point are ignored, if >= 0",
		           -1);
		r.property("SmoothVelocity", mSmooth, setterNotify(mSmooth, &TrajectoryVisualizationBase::update, this), "smoothing (lowpass) of velocity [0.0=none, 1.0=extreme]",
		           0.0, PropertyHints::minimum(0.0) | PropertyHints::maximum(1.0));
	}

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mLineObject = new LineStripObject(mgr,node);
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		if (mLineObject != NULL)
			mLineObject->setVisible(enabled);
	}

	void setColormap(const ColormapProperty& colormap)
	{
		mColorMap = colormap;
		mColorMap.setFilter(ColormapProperty::filterContinuous);

		update();
	}

	void update() {
		if(!mLineObject)
			return;

		mLineObject->clear();
		mLineObject->setLineWidth(mLineWidth);
		if (mTrajectory.size() == 0)
			return;

		Ogre::ColourValue color(1.0, 0.0, 0.0, 1.0);
		mLineObject->begin();
		const ContinuousColormap& colormap = dynamic_cast<const ContinuousColormap&>(mColorMap.getColormap());

		std::size_t start = 0;
		std::size_t end = mTrajectory.size()-1;

		if (mStartIndex >= 0)
			start = mStartIndex = mStartIndex % mTrajectory.size();

		if (mEndIndex >= 0)
			end = mEndIndex = mEndIndex % mTrajectory.size();

		double velocity = 0;
		std::size_t i=start;
		for(; (i<=end) && (i+1 < mTrajectory.size()); ++i)
		{
			std::pair<Ogre::Vector3, float> pd = getPoint(mTrajectory[i], mTrajectory[i+1]);

			if (mMinVelocity < mMaxVelocity)
			{
				double seconds = (mTrajectory[i+1].t - mTrajectory[i].t).totalMicroseconds() / 1e6;
				double v = pd.second / seconds;
				velocity = mSmooth * velocity + (1.f - mSmooth) * v;
				Color::RGB miraColor =  colormap.getf((velocity-mMinVelocity)/(mMaxVelocity-mMinVelocity));
				color = Ogre::ColourValue(miraColor.r, miraColor.g, miraColor.b, 1.0);
			}
			mLineObject->point(pd.first, color);
		}
		// for trajectory's final point we can't determine velocity from subsequent point
		// if it needs drawing, draw with previous point's color (or default color red)
		if (i<=end)
		{
			Ogre::Vector3 p = getPoint(mTrajectory[i], mTrajectory[i]).first;
			mLineObject->point(p, color);
		}
		mLineObject->end();
	}

	virtual std::pair<Ogre::Vector3, float> getPoint(const typename Trajectory::Sample & sample1,
													 const typename Trajectory::Sample & sample2)
	{
		Ogre::Vector3 p1 = OgreUtils::toOgreVector(sample1.p);
		Ogre::Vector3 p2 = OgreUtils::toOgreVector(sample2.p);
		float d = (Point3f(p1.x, p1.y, p1.z) - Point3f(p2.x, p2.y, p2.z)).norm();
		return std::pair<Ogre::Vector3, float>(p1, d);
	}

	void dataChanged(ChannelRead<Trajectory> trajectory)
	{
		mTrajectory = *trajectory;
		update();
	}

private:
	Trajectory mTrajectory;
	float mMinVelocity;
	float mMaxVelocity;
	ColormapProperty mColorMap;
	LineStripObject* mLineObject;
	float mLineWidth;
	int mStartIndex;
	int mEndIndex;
	float mSmooth;
};

///////////////////////////////////////////////////////////////////////////////

class Trajectory2Visualization : public TrajectoryVisualizationBase<PoseTrajectory2>
{
	MIRA_META_OBJECT(Trajectory2Visualization,
			("Name", "Trajectory2")
			("Description", "Visualization of a trajectory of 2D poses")
			("Category", "Robot Datatypes"))

public:

	Trajectory2Visualization() :
		mTimeToZ(0.f)
	{
		mStartTime = Time::invalid();
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		TrajectoryVisualizationBase<PoseTrajectory2>::reflect(r);
		r.property("TimeToZ", mTimeToZ, setterNotify(mTimeToZ, &Trajectory2Visualization::update, (TrajectoryVisualizationBase<PoseTrajectory2>*)this), "Factor to draw progressing time as height [cm/s]",
				   0.f);
	}

	virtual std::pair<Ogre::Vector3, float> getPoint(const PoseTrajectory2::Sample & sample1,
	                                                 const PoseTrajectory2::Sample & sample2)
	{
		Ogre::Vector3 p = OgreUtils::toOgreVector(sample1.p);

		if (mStartTime.isValid())
		{
			p.z = (sample1.t - mStartTime).totalSeconds() * mTimeToZ * 0.01;
		}
		else
		{
			mStartTime = sample1.t;
			p.z = 0.f;
		}

		float d = hypotf(sample1.p.x() - sample2.p.x(), sample1.p.y() - sample2.p.y());

		return std::pair<Ogre::Vector3, float>(p, d);
	}

private:

	Time mStartTime;
	float mTimeToZ;
};

class Trajectory3Visualization : public TrajectoryVisualizationBase<PoseTrajectory3>
{
	MIRA_META_OBJECT(Trajectory3Visualization,
			("Name", "Trajectory3")
			("Description", "Visualization of a trajectory of 3D poses")
			("Category", "Robot Datatypes"))

	virtual std::pair<Ogre::Vector3, float> getPoint(const PoseTrajectory3::Sample & sample1,
	                                                 const PoseTrajectory3::Sample & sample2)
	{
		Ogre::Vector3 p = OgreUtils::toOgreVector(sample1.p);
		float d = (Point3f(sample1.p.x(), sample1.p.y(), sample1.p.z())
				 - Point3f(sample2.p.x(), sample2.p.y(), sample2.p.z())).norm();
		return std::pair<Ogre::Vector3, float>(p, d);
	}
};


///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Trajectory2Visualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Trajectory3Visualization,
                         mira::Visualization3D);
