/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PreferredDirectionTask.h
 *    Direction task with a given preferred direction
 *
 * @author Tim Langner
 * @date   2011/05/30
 */

#ifndef _MIRA_PREFERREDDIRECTIONTASK_H_
#define _MIRA_PREFERREDDIRECTIONTASK_H_

#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Sub task that has the purpose of limiting the direction the robot drives.
 */
class MIRA_NAVIGATION_EXPORT PreferredDirectionTask : public SubTask
{
	MIRA_OBJECT(PreferredDirectionTask);
public:

	/// There are three possible directions that can be allowed
	enum Direction
	{
		BOTH = 0,
		BACKWARD = -1,
		FORWARD = 1
	};

	/**
	 * Creates a direction task with a given allowed direction and a cost for
	 * not driving in that direction.
	 * If cost is 0 also driving in the wrong direction is allowed.
	 * If cost is 1 driving in wrong direction is totally forbidden.
	 * @param[in] iDirection          The requested direction.
	 * @param[in] iWrongDirectionCost The costs.
	 */
	PreferredDirectionTask(Direction iDirection = FORWARD, 
	                       float iWrongDirectionCost = 1.0f) :
		direction(iDirection),
		wrongDirectionCost(iWrongDirectionCost) {}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		SubTask::reflect(r);
		r.property("Direction", direction, "The preferred direction.");
		r.property("WrongDirectionCost", wrongDirectionCost,
		           "Costs for driving into not-preferred direction.");
	}

public:

	/// The preferred direction.
	Direction direction;

	/// Costs for driving into not-preferred direction.
	float wrongDirectionCost;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
