/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PersonApproachTask.h
 *    Task for driving to a certain distance to a person.
 *
 * @author Erik Einhorn
 * @date   2012/02/17
 */

#ifndef _MIRA_PERSONAPPROACHTASK_H_
#define _MIRA_PERSONAPPROACHTASK_H_

#include <transform/Pose.h>

#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Task for approaching a person in a certain distance.
 */
class MIRA_NAVIGATION_EXPORT PersonApproachTask : public SubTask
{
	MIRA_OBJECT(PersonApproachTask);
public:

	/// Creates a task with zero min and max distance and zero tolerance.
	PersonApproachTask() :
		minDistance(0.0f),
		maxDistance(0.0f),
		orientationTolerance(0.0f)
	{}

	/**
	 * @brief Creates a task with given pose and a tolerance.
	 * @param[in] minDist   The minimum distance to the person in [m].
	 * @param[in] maxDist   The maximum distance to the person in [m].
	 * @param[in] orientTol The allowed tolerance for orientation in [rad].
	 */
	PersonApproachTask(float minDist, float maxDist, float orientTol) :
		minDistance(minDist),
		maxDistance(maxDist),
		orientationTolerance(orientTol)
	{}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		SubTask::reflect(r);
		r.property("MinDistance", minDistance,
		           "The minimum distance to the person in [m].");
		r.property("MaxDistance", maxDistance,
		           "The maximum distance to the person in [m].");
		r.property("OrientationTolerance", orientationTolerance,
		           "The allowed tolerance for orientation in [rad].");
	}

public:
	/// The minimum distance to the person in [m].
	float minDistance;

	/// The maximum distance to the person in [m].
	float maxDistance;

	/// The allowed tolerance for orientation in [rad].
	float orientationTolerance;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
