/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OrientationTask.h
 *    Orientation task with a given angle and a variance.
 *
 * @author Tim Langner
 * @date   2011/04/18
 */

#ifndef _MIRA_ORIENTATIONTASK_H_
#define _MIRA_ORIENTATIONTASK_H_

#include <math/Angle.h>

#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Sub task that has the purpose to reach a given orientation with at least
 * the given variance.
 */
class MIRA_NAVIGATION_EXPORT OrientationTask : public SubTask
{
	MIRA_OBJECT(OrientationTask);
public:

	/// Create orientation task with orientation 0 and variance 0.
	OrientationTask() :
		phi(0), tolerance(0), frameID("/GlobalFrame") {}

	/**
	 * @brief Create orientation task with given orientation and variance
	 * @param[in] p The requested orientation in [rad].
	 * @param[in] t The requested variance in [rad].
	 * @param[in] f The frame id the angle is related to.
	 */
	OrientationTask(float p, float t, const std::string& f = "/GlobalFrame") :
		phi(p), tolerance(t), frameID(f)
	{}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		SubTask::reflect(r);
		r.property("Orientation", phi, "The requested orientation.");
		r.property("Tolerance", tolerance, "The requested tolerance.");
		r.property("FrameID", frameID,
		           "The frame the angle is related to.", "/GlobalFrame");
	}

	/// Calculates if the task is reached for the current orientation p
	bool reached(float p) const
	{
		return std::abs(phi.smallestDifferenceValue(Anglef(p))) <= tolerance.rad();
	}

public:
	/// The requested orientation.
	Anglef phi;

	/// The requested tolerance.
	Anglef tolerance;

	/// The frame the angle is related to
	std::string frameID;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
