/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FramePublisherView.C
 *    Implementation of FramePublisherView.h.
 *
 * @author Erik Einhorn
 * @date   2014/04/03
 */

#include <QTimer>

#include <math/Angle.h>
#include <math/YawPitchRoll.h>
#include <transform/Pose.h>

#include "private/FramePublisherView.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

FramePublisherView::FramePublisherView() :
	ui(NULL)
{
}

FramePublisherView::~FramePublisherView()
{
}

QWidget* FramePublisherView::createPartControl()
{
	ui = new UI(this);
	return ui;
}

static QDoubleSpinBox* initSpinBox(QDoubleSpinBox* sb)
{
	sb->setMinimum(-10.0e8);
	sb->setMaximum(10.0e8);
	sb->setSingleStep(0.1);
	sb->setDecimals(3);
	return sb;
}

static QDoubleSpinBox* initSpinBoxAngle(QDoubleSpinBox* sb)
{
	sb->setMinimum(-180);
	sb->setMaximum(180);
	return sb;
}

FramePublisherView::UI::UI(FramePublisherView* parent) :
	QWidget(parent),
	authority("/", "FramePublisherView", Authority::ANONYMOUS | Authority::INTERNAL),
	mParent(parent)
{
	mLayout = new QFormLayout(this);


	mSbX = initSpinBox(new QDoubleSpinBox(this));
	mSbY = initSpinBox(new QDoubleSpinBox(this));
	mSbZ = initSpinBox(new QDoubleSpinBox(this));
	mSbYaw   = initSpinBoxAngle(new QDoubleSpinBox(this));
	mSbPitch = initSpinBoxAngle(new QDoubleSpinBox(this));
	mSbRoll  = initSpinBoxAngle(new QDoubleSpinBox(this));

	connect(mSbX, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbY, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbZ, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbYaw,   SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbPitch, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbRoll,  SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));

	mSelectFrame = new QComboBox(this);
	connect(mSelectFrame, SIGNAL(activated(int)), this, SLOT(updateControls()));
	mSelectFrame->setDuplicatesEnabled(false);

	mLayout->addRow("Frame ID:", mSelectFrame);
	mLayout->addRow("X:", mSbX);
	mLayout->addRow("Y:", mSbY);
	mLayout->addRow("Z:", mSbZ);
	mLayout->addRow("Yaw:", mSbYaw);
	mLayout->addRow("Pitch:", mSbPitch);
	mLayout->addRow("Roll:", mSbRoll);

	QTimer* timer = new QTimer(this);
	timer->setInterval(2000);
	timer->setSingleShot(false);

	connect(timer, SIGNAL(timeout()), this, SLOT(updateControls()));
	timer->start();
}

void FramePublisherView::UI::updateControls()
{
	updateFrames();

	std::string frameID = mSelectFrame->currentText().toStdString();
	if(frameID.empty()) {
		// set default values in UI for empty frame
		setEdits(0.f, 0.f, 0.f, 0.f, 0.f, 0.f,
		         false, false, false,false, false, false);	
		return;
	}

	try {
		try {
			auto ch = authority.subscribe<Pose3>(frameID);
			auto r = ch.read();

			Pose3 tf = r->value();
			
			setEdits(tf.x(), tf.y(), tf.z(), rad2deg(tf.yaw()), rad2deg(tf.pitch()), rad2deg(tf.roll()),
			         true, true, true, true, true, true);

		} catch(XBadCast&) {
			auto ch = authority.subscribe<Pose2>(frameID);
			auto r = ch.read();

			Pose2 tf = r->value();

			mSbZ->setEnabled(false);
			mSbPitch->setEnabled(false);
			mSbRoll->setEnabled(false);
			setEdits(tf.x(), tf.y(), 0.f, rad2deg(tf.phi()), 0.f, 0.f,
			         true, true, false, true, false, false);
		}
	}
	catch(...) {
		setEdits(0.f, 0.f, 0.f, 0.f, 0.f, 0.f,
		         false, false, false,false, false, false);
	}
}

void FramePublisherView::UI::updateFrames()
{
	auto nodes = MIRA_FW.getTransformer()->getNodes();	
	foreach(TransformerBase::AbstractNodePtr node, nodes) {
		QString frameID = QString::fromStdString(node->getID());
		if (mSelectFrame->findText(frameID, Qt::MatchExactly) < 0)
			mSelectFrame->addItem(frameID);
	}
}

void FramePublisherView::UI::updateFrame()
{
	std::string frameID = mSelectFrame->currentText().toStdString();
	if(frameID.empty())
		return;

	auto ch = authority.subscribe<void>(frameID);
	Time timestamp = Time::now();

	try {
		auto r = ch.read();
		timestamp = r->timestamp + Duration::microseconds(1);
	} catch(...) {}

	// extract the values into a transform

	if (mSbZ->isEnabled()) {
		Pose3 tf;

		tf.t.x() = mSbX->value();
		tf.t.y() = mSbY->value();
		tf.t.z() = mSbZ->value();

		float yaw = deg2rad(mSbYaw->value());
		float pitch = deg2rad(mSbPitch->value());
		float roll = deg2rad(mSbRoll->value());

		tf.r = quaternionFromYawPitchRoll(yaw,pitch,roll);

		authority.publishTransform<Pose3>(frameID, tf, timestamp);
	} else {
		Pose2 tf;

		tf.t.x() = mSbX->value();
		tf.t.y() = mSbY->value();

		tf.r = Pose2::RotationType(deg2rad(mSbYaw->value()));

		authority.publishTransform<Pose2>(frameID, tf, timestamp);
	}
}

void FramePublisherView::UI::setEdits(float x, float y, float z, float yaw, float pitch, float roll,
                                      bool enableX, bool enableY, bool enableZ,
                                      bool enableYaw, bool enablePitch, bool enableRoll)
{
	blockEditSignals(true);

	mSbX->setValue(x);
	mSbY->setValue(y);
	mSbZ->setValue(z);

	mSbYaw->setValue(yaw);
	mSbPitch->setValue(pitch);
	mSbRoll->setValue(roll);

	mSbX->setEnabled(enableX);
	mSbY->setEnabled(enableY);
	mSbZ->setEnabled(enableZ);

	mSbYaw->setEnabled(enableYaw);
	mSbPitch->setEnabled(enablePitch);
	mSbRoll->setEnabled(enableRoll);
	
	blockEditSignals(false);
}

void FramePublisherView::UI::blockEditSignals(bool block)
{
	mSbX->blockSignals(block);
	mSbY->blockSignals(block);
	mSbZ->blockSignals(block);

	mSbYaw->blockSignals(block);
	mSbPitch->blockSignals(block);
	mSbRoll->blockSignals(block);
}


///////////////////////////////////////////////////////////////////////////////

}
MIRA_CLASS_SERIALIZATION(mira::FramePublisherView, mira::ViewPart);
