/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PointVectorVisualization.C
 *    Visualization of vector of points.
 *
 * @author Christian Reuther
 * @date   2015/10/27
 */

#include <OGRE/OgreSceneManager.h>

#include <transform/Pose.h>

#include <serialization/Serialization.h>
#include <serialization/GetterSetter.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <math/EigenFormat.h>
#include <visualization/Visualization3D.h>
#include <visualization/3d/DynamicPoints.h>
#include <widgets/OgreUtils.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
struct PointTypeDimensionTrait { static const int dims = -1; };

template <> struct PointTypeDimensionTrait<Point2f> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Point2d> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Eigen::Vector2f> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Eigen::Vector2d> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Point3f> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Point3d> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Eigen::Vector3f> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Eigen::Vector3d> { static const int dims = 3; };

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
class PointVectorVisualizationBase : public Visualization3D
{
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	typedef std::vector<PointType> PointVector;

public:
	PointVectorVisualizationBase()
	{
		mPointsObject = NULL;
		mNode = NULL;
		mPointsColor = Ogre::ColourValue::Blue;
		mDataChannel.setDataChangedCallback(boost::bind(&PointVectorVisualizationBase::dataChangedCallback, this, _1));
	}

	virtual ~PointVectorVisualizationBase()
	{
		delete mPointsObject;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Visualization3D::reflect(r);
		channelProperty(r, "PointVector", mDataChannel, "The channel to be visualized");

		r.property("Color", mPointsColor, setter<Ogre::ColourValue>(&PointVectorVisualizationBase::setColor, this),
				"Color of the points", Ogre::ColourValue::Blue);
	}

public:
	virtual void setupScene(IVisualization3DSite* site)
	{
		mPointsObject = new DynamicPoints();
		mNode = site->getSceneManager()->getRootSceneNode()->createChildSceneNode();
		mNode->attachObject(mPointsObject);
	}

	virtual Ogre::SceneNode* getNode()
	{
		return mNode;
	}

	void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		if(mPointsObject)
			mPointsObject->setVisible(enabled);
	}

	virtual DataConnection getDataConnection()
	{
		return DataConnection(mDataChannel);
	}

	void setColor(const Ogre::ColourValue& c)
	{
		mPointsColor = c;
		if(mPointsObject)
			mPointsObject->setColor(c);
	}

protected:
	virtual void dataChanged(ChannelRead<PointVector> data)
	{
		if(!mPointsObject)
			return;

		const int dim = PointTypeDimensionTrait<PointType>::dims;

		mPointsObject->clear();
		mPointsObject->update();
		const auto& vec = data->value();
		foreach(const auto& p, vec)
			mPointsObject->addPoint(p(0), p(1), dim >= 3 ? p(2) : 0.0, mPointsColor);
		mPointsObject->update();
	}

	virtual void update(Duration dt)
	{
		if (!mDataChannel.isValid())
			return;

		mParentTransform = getAuthority().template getTransform<Pose3>(mDataFrameID, getSite()->getCameraFrame());
		OgreUtils::setTransform(mNode, mParentTransform);
	}

protected:
	void dataChangedCallback(ChannelRead<PointVector> data)
	{
		mDataFrameID = data->frameID;
		mDataTimestamp = data->timestamp;
		dataChanged(data);
	}

protected:
	Ogre::SceneNode* mNode;
	DynamicPoints* mPointsObject;

	ChannelProperty<PointVector> mDataChannel;

	Ogre::ColourValue mPointsColor;

	std::string mDataFrameID;
	Time mDataTimestamp;
	Pose3 mParentTransform;
};

///////////////////////////////////////////////////////////////////////////////

class PointVectorVisualization2d : public PointVectorVisualizationBase<Point2d>
{
	MIRA_META_OBJECT(PointVectorVisualization2d,
		("Name", "PointVector2d")
		("Description", "Visualization of a vector of Point2d")
		("Category", "Geometry"))
};

class PointVectorVisualization3d : public PointVectorVisualizationBase<Point3d>
{
	MIRA_META_OBJECT(PointVectorVisualization3d,
		("Name", "PointVector3d")
		("Description", "Visualization of a vector of Point3d")
		("Category", "Geometry"))
};

class PointVectorVisualizationVector2d : public PointVectorVisualizationBase<Eigen::Vector2d>
{
	MIRA_META_OBJECT(PointVectorVisualizationVector2d,
		("Name", "PointVectorVector2d")
		("Description", "Visualization of a vector of Eigen::Vector2d")
		("Category", "Geometry"))
};

class PointVectorVisualizationVector3d : public PointVectorVisualizationBase<Eigen::Vector3d>
{
	MIRA_META_OBJECT(PointVectorVisualizationVector3d,
		("Name", "PointVectorVector3d")
		("Description", "Visualization of a vector of Eigen::Vector3d")
		("Category", "Geometry"))
};

//////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::PointVectorVisualization2d, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PointVectorVisualization3d, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PointVectorVisualizationVector2d, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PointVectorVisualizationVector3d, mira::Visualization3D);
