/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PointVectorVisualization.C
 *    Visualization of vectors of points in 2D space.
 *
 * @author Christian Reuther
 * @date   2014/08/27
 */

#include <Eigen/Eigen>
#include <Eigen/Eigenvalues>

#include <QGraphicsScene>
#include <QGraphicsPathItem>
#include <QVector>
#include <QDebug>

#include <serialization/Serialization.h>
#include <serialization/adapters/Qt/QColor>
#include <visualization/Visualization2D.h>
#include <visualization/2d/PointVectorObject.h>
#include <widgets/QtUtils.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename Point>
class PointVectorVisualization2D :  public Visualization2D {
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	typedef std::vector<Point> PointVector;

public:
	PointVectorVisualization2D() : mItem(NULL) {
		mDataChannel.setDataChangedCallback(boost::bind(&PointVectorVisualization2D::onDataChanged, this, _1));

		// Create this here, as we want the reflect method to be able to access its members
		mItem = new PointVectorObject<Point>();
		mItem->setVisible(false);
	}

	virtual ~PointVectorVisualization2D() {
		auto site = this->getSite();
		// For some reason it is possible that the view gets destructed and re-constructed
		// when miracenter starts, so we need to make sure that we can access the items
		if(!mItem || !site || !site->getSceneManager())
			return;

		site->getSceneManager()->removeItem(mItem);
		delete mItem;
	}

	template <typename Reflector> void reflect(Reflector& r) {
		Visualization2D::reflect(r);
		channelProperty(r, "PoseVector", mDataChannel, "Pose vector channel to be visualized");

		r.property("PointSize", getter<qreal>(&PointVectorVisualization2D::pointSize, this),
			setter<qreal>(&PointVectorVisualization2D::setPointSize, this), "The width of a path point [m]", 0.025f);
		r.property("Color", getter<QColor>(&PointVectorVisualization2D::color, this),
			setter<QColor>(&PointVectorVisualization2D::setColor, this), "The color to use for painting the path", QColor(255, 0, 0, 180));
	}

public:
	void setPointSize(qreal size) {
		if(!mItem)
			return;
		mItem->setWidth(size);
	}

	qreal pointSize() const {
		return !mItem ? 0.0 : mItem->width();
	}

	void setColor(QColor color) {
		mItem->setColor(color);
	}

	QColor color() const {
		return !mItem ? Qt::transparent : mItem->color();
	}

public:
	virtual Visualization::DataConnection getDataConnection() {
		return Visualization::DataConnection(mDataChannel);
	}

	virtual void setEnabled(bool enabled) {
		Visualization2D::setEnabled(enabled);
		mItem->setVisible(enabled);
	}

	virtual void setupScene(IVisualization2DSite* site) {
		QGraphicsScene* mgr = site->getSceneManager();
		mgr->addItem(mItem);
	}

	virtual QGraphicsItem* getItem()
	{
		return mItem;
	}

	void onDataChanged(ChannelRead<PointVector> data) {
		mItem->setData(data->value());

		RigidTransform2f d = this->getAuthority().template getTransform<RigidTransform2f>(data->frameID, data->timestamp,
				this->getSite()->getCameraFrame(), Time::now(), this->getSite()->getFixedFrame(), LinearInterpolatorNearestNeighbourExtrapolator());
		QtUtils::setTransform(mItem, d);
	}

protected:
	ChannelProperty<PointVector> mDataChannel;
	PointVectorObject<Point>* mItem;
};

///////////////////////////////////////////////////////////////////////////////

class Point2fVectorVisualization2D : public PointVectorVisualization2D<Point2f>
{
	MIRA_META_OBJECT(Point2fVectorVisualization2D,
		("Name", "Point2f Vector")
		("Description", "Visualization of a std::vector<Point2f>")
		("Category", "Pose"))
};

class Point3fVectorVisualization2D : public PointVectorVisualization2D<Point3f>
{
	MIRA_META_OBJECT(Point3fVectorVisualization2D,
		("Name", "Point3f Vector")
		("Description", "Visualization of a std::vector<Point3f>")
		("Category", "Pose"))
};

//////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::Point2fVectorVisualization2D, mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3fVectorVisualization2D, mira::Visualization2D);
