/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file STLMeshLoader.C
 *     Loads meshs from stl files.
 *
 * @author Tim Langner
 * @date   2011/12/09
 */

#include <visualization/3d/MeshLoader.h>

#include <fstream>

#include <OGRE/Ogre.h>

#include <utils/Buffer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class STLMeshLoader : public MeshLoader
{
	MIRA_META_OBJECT(STLMeshLoader,
	                ("Extension", "Stereolithography,.stl"))
public:

	Ogre::MeshPtr loadMesh(const Path& filename);
};

///////////////////////////////////////////////////////////////////////////////

struct STLTriangle
{
	Ogre::Vector3 normal;
	Ogre::Vector3 vertex[3];
};

struct STLMesh
{
	std::vector<STLTriangle> triangles;
};

Ogre::Vector3 readVector(uint8*& ioPtr)
{
	Ogre::Vector3 v;
	v.x = *reinterpret_cast<float*>(ioPtr);
	ioPtr += 4;
	v.y = *reinterpret_cast<float*>(ioPtr);
	ioPtr += 4;
	v.z = *reinterpret_cast<float*>(ioPtr);
	ioPtr += 4;
	return v;
}

void calculateUV(const Ogre::Vector3& vec, float& u, float& v)
{
	Ogre::Vector3 pos(vec);
	pos.normalise();
	u = acos(pos.y / pos.length());

	float val = pos.x / (sin(u));
	v = acos(val);

	u /= Ogre::Math::PI;
	v /= Ogre::Math::PI;
}

void loadMeshASCII(STLMesh& oMesh, const Path& filename)
{
	std::ifstream is;
	is.open(filename.string().c_str());
	std::string solidToken, nameToken,
		facetToken, normalToken, outerToken, loopToken,
		vertexToken, endLoopToken, endFacetToken;
	// read solid name
	is >> solidToken >> nameToken;
	if (solidToken != "solid")
		MIRA_THROW(XIO, "Invalid STL ASCII format (missing 'solid name' header)");

	uint32 vertexCount = 0;
	STLTriangle tri;
	while (!is.eof())
	{
		// read facet normal or endsolid
		is >> facetToken;
		// if endsolid we reached end of file
		if (facetToken == "endsolid")
			break;
		is >> normalToken;
		if (facetToken != "facet" || normalToken != "normal")
			MIRA_THROW(XIO, "Invalid STL ASCII format (expected 'facet normal nx ny nz')");

		// read normal
		is >> tri.normal.x >> tri.normal.y >> tri.normal.z;
		// read outer loop
		is >> outerToken >> loopToken;
		if (outerToken != "outer" || loopToken != "loop")
			MIRA_THROW(XIO, "Invalid STL ASCII format (expected 'outer loop')");

		// read vertex vx vy vz
		for(int i=0; i<3; ++i)
		{
			is >> vertexToken >> tri.vertex[i].x >> tri.vertex[i].y >> tri.vertex[i].z;
			if (vertexToken != "vertex")
				MIRA_THROW(XIO, "Invalid STL ASCII format (expected 'vertex vx vy vz')");
		}

		is >> endLoopToken >> endFacetToken;
		if (endLoopToken != "endloop" || endFacetToken != "endfacet")
			MIRA_THROW(XIO, "Invalid STL ASCII format (expected 'endloop endfacet')");

		oMesh.triangles.push_back(tri);
	}
}

void loadMeshBinary(STLMesh& oMesh, uint8* ptr)
{
	uint32 numberOfTriangles = *reinterpret_cast<uint32*>(ptr);
	ptr += 4;
	oMesh.triangles.resize(numberOfTriangles);

	for (uint32 t = 0; t < numberOfTriangles; ++t)
	{
		oMesh.triangles[t].normal = readVector(ptr);
		for(int i=0; i<3; ++i)
			oMesh.triangles[t].vertex[i] = readVector(ptr);

		// skip 2 byte attribute
		ptr += 2;
	}
}

Ogre::MeshPtr STLMeshLoader::loadMesh(const Path& filename)
{
	std::ifstream is;
	is.open(filename.string().c_str());

	// get length of file:
	is.seekg (0, std::ios::end);
	std::streamsize length = is.tellg();
	is.seekg (0, std::ios::beg);

	Buffer<uint8> buffer(length);
	is.read((char*)buffer.data(), length);
	is.close();

	// check format (ascii or binary)
	// Binary format:
	// UINT8[80] – Header
	// UINT32 – Number of triangles
	//
	// foreach triangle
	// REAL32[3] – Normal vector
	// REAL32[3] – Vertex 1
	// REAL32[3] – Vertex 2
	// REAL32[3] – Vertex 3
	// UINT16 – Attribute byte count
	// end
	//
	// ASCII format
	// solid name
	//
	// foreach triangle
	// facet normal ni nj nk
	// outer loop
	// vertex v1x v1y v1z
	// vertex v2x v2y v2z
	// vertex v3x v3y v3z
	// endloop
	// endfacet
	//
	// endsolid name
	// The inventors of STL put a lot of efford into the format to make it
	// nearly impossible to easily determine if the format of a file is ASCII or
	// binary.
	// ASCII files start with "solid name" but we can not be sure that when
	// a file starts with "solid" its format is ASCII since the 80 byte header
	// of a binary file can also start with "solid" (yeah great idea STL creators)

	STLMesh stlMesh;

	// assume we have a binary file so skip 80 bytes header
	// read triangle count and do a sanity check
	uint8* ptr = buffer.data()+80;
	uint32 numberOfTriangles = *reinterpret_cast<uint32*>(ptr);
	// sanity check (file size - header) must equal (3*4 + 3*4 + 3*4 + 2) * triangleCount
	if (uint32(length-84) == numberOfTriangles * (4*3*4+2))
	{
		// Binary format
		loadMeshBinary(stlMesh, ptr);
	}
	else
	{
		// ASCII format
		loadMeshASCII(stlMesh, filename);
	}

	if(stlMesh.triangles.empty())
		MIRA_THROW(XIO, "Cannot read mesh from '" << filename << "' or the mesh is empty.");

	std::string group = "MIRAMeshs";

	// If the resource group does not exist, create it to prevent an exception.
	Ogre::ResourceGroupManager& mgr = Ogre::ResourceGroupManager::getSingleton();
	if (!mgr.resourceGroupExists(group))
		mgr.createResourceGroup(group);

	Ogre::MeshPtr mesh = Ogre::MeshManager::getSingleton().createManual( filename.string(), group );

	// fill the ogre mesh using our stl mesh data
	Ogre::SubMesh* subMesh = mesh->createSubMesh();

	Ogre::VertexData* data = new Ogre::VertexData();
	mesh->sharedVertexData = data;
	data->vertexCount = stlMesh.triangles.size()*3;

	Ogre::VertexDeclaration* decl = data->vertexDeclaration;
	size_t offset = 0;
	decl->addElement(0, offset, Ogre::VET_FLOAT3, Ogre::VES_POSITION);
	offset += Ogre::VertexElement::getTypeSize(Ogre::VET_FLOAT3);
	decl->addElement(0, offset, Ogre::VET_FLOAT3, Ogre::VES_NORMAL);
	offset += Ogre::VertexElement::getTypeSize(Ogre::VET_FLOAT3);
	decl->addElement(0, offset, Ogre::VET_FLOAT2, Ogre::VES_TEXTURE_COORDINATES);

	Ogre::HardwareVertexBufferSharedPtr vbuf = Ogre::HardwareBufferManager::getSingleton().createVertexBuffer(
			decl->getVertexSize(0),
			data->vertexCount,
			Ogre::HardwareBuffer::HBU_STATIC_WRITE_ONLY
	 );

	float* vertexData = static_cast<float*>(vbuf->lock(Ogre::HardwareVertexBuffer::HBL_DISCARD));

	Ogre::HardwareIndexBufferSharedPtr ibuf = Ogre::HardwareBufferManager::getSingleton().createIndexBuffer(
			Ogre::HardwareIndexBuffer::IT_16BIT,
			data->vertexCount,
			Ogre::HardwareBuffer::HBU_STATIC_WRITE_ONLY
	 );

	uint16* indexData = static_cast<uint16*>(ibuf->lock(Ogre::HardwareIndexBuffer::HBL_DISCARD));

	Ogre::AxisAlignedBox aabb;

	for(std::size_t t=0; t<stlMesh.triangles.size(); ++t)
	{
		Ogre::Vector3 normal = stlMesh.triangles[t].normal;
		if (normal.squaredLength() < 0.001f)
		{
			Ogre::Vector3 side1 = stlMesh.triangles[t].vertex[0] - stlMesh.triangles[t].vertex[1];
			Ogre::Vector3 side2 = stlMesh.triangles[t].vertex[1] - stlMesh.triangles[t].vertex[2];
			normal = side1.crossProduct(side2);
		}

		for(int i=0; i<3; ++i)
		{
			vertexData[0] = stlMesh.triangles[t].vertex[i].x;
			vertexData[1] = stlMesh.triangles[t].vertex[i].y;
			vertexData[2] = stlMesh.triangles[t].vertex[i].z;
			vertexData[3] = normal.x;
			vertexData[4] = normal.y;
			vertexData[5] = normal.z;

			float u,v;
			calculateUV(stlMesh.triangles[t].vertex[i], u, v);
			vertexData[6] = u;
			vertexData[7] = v;

			vertexData+=8;


			*indexData = t*3+i;
			indexData++;

			aabb.merge(stlMesh.triangles[t].vertex[i]);
		}
	}

	vbuf->unlock();
	ibuf->unlock();

	Ogre::VertexBufferBinding* bind = mesh->sharedVertexData->vertexBufferBinding;
	bind->setBinding(0, vbuf);

	/// Set parameters of the submesh
	subMesh->useSharedVertices = true;
	subMesh->indexData->indexBuffer = ibuf;
	subMesh->indexData->indexCount = data->vertexCount;
	subMesh->indexData->indexStart = 0;

	mesh->_setBounds(aabb);

	Ogre::Vector3 s = aabb.getSize();
	mesh->_setBoundingSphereRadius(std::max(s.x, std::max(s.y, s.z))/2.0f);

	mesh->load();

	return mesh;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::STLMeshLoader, mira::MeshLoader)
