/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImageObject.C
 *    Implementation of ImageObject.
 *
 * @author Tim Langner
 * @date   2011/01/06
 */

#include <utils/ToString.h>

#include <visualization/3d/ImageObject.h>

#include <boost/scoped_array.hpp>

#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreVector3.h>
#include <OGRE/OgreQuaternion.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreHardwarePixelBuffer.h>
#include <OGRE/OgreMeshManager.h>
#include <OGRE/OgreEntity.h>
#include <OGRE/OgreImage.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ImageObject::ImageObject(bool mirrorY, Ogre::SceneManager* sceneManager,
                         Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mPlane(NULL)
{
	mMaterial = Ogre::MaterialManager::getSingleton().create(
			"ImageObject"+toString(this)+"/Material", "MIRAMaterials");
	mMaterial->setReceiveShadows(false);
	mMaterial->getTechnique(0)->setLightingEnabled(false);
	mMaterial->getTechnique(0)->setCullingMode(Ogre::CULL_NONE);
	mMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
	mMaterial->setDepthWriteEnabled(false);

	Ogre::Vector3 p1(0.0f, 0.0f, 0.0f);
	Ogre::Vector3 p2(1.0f, 1.0f, 0.0f);
	Ogre::Vector3 p3(0.0f, 1.0f, 0.0f);
	Ogre::Vector3 p4(1.0f, 0.0f, 0.0f);
	if (mirrorY)
	{
		p1 = Ogre::Vector3(0.0f, 1.0f, 0.0f);
		p2 = Ogre::Vector3(1.0f, 0.0f, 0.0f);
		p3 = Ogre::Vector3(0.0f, 0.0f, 0.0f);
		p4 = Ogre::Vector3(1.0f, 1.0f, 0.0f);
	}

	mPlane = mSceneManager->createManualObject("ImageObject"+toString(this));
	mPlane->begin(mMaterial->getName(),	Ogre::RenderOperation::OT_TRIANGLE_LIST);
	mPlane->position(p1);
	mPlane->textureCoord(0.0f, 0.0f);
	mPlane->normal(0.0f, 0.0f, 1.0f);
	mPlane->position(p2);
	mPlane->textureCoord(1.0f, 1.0f);
	mPlane->normal(0.0f, 0.0f, 1.0f);
	mPlane->position(p3);
	mPlane->textureCoord(0.0f, 1.0f);
	mPlane->normal(0.0f, 0.0f, 1.0f);

	mPlane->position(p1);
	mPlane->textureCoord(0.0f, 0.0f);
	mPlane->normal(0.0f, 0.0f, 1.0f);
	mPlane->position(p4);
	mPlane->textureCoord(1.0f, 0.0f);
	mPlane->normal(0.0f, 0.0f, 1.0f);
	mPlane->position(p2);
	mPlane->textureCoord(1.0f, 1.0f);
	mPlane->normal(0.0f, 0.0f, 1.0f);
	mPlane->end();

	mNode->attachObject(mPlane);
}

ImageObject::~ImageObject()
{
	if ( mPlane != NULL )
	{
		mNode->detachObject(mPlane);
		mSceneManager->destroyManualObject(mPlane);
	}
	if (!mMaterial.isNull())
		Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());
}

void ImageObject::setAlpha(float alpha)
{
	Ogre::Pass* pass = mMaterial->getTechnique(0)->getPass(0);
	Ogre::TextureUnitState* tex = NULL;
	if (pass->getNumTextureUnitStates() > 0)
		tex = pass->getTextureUnitState(0);
	else
		tex = pass->createTextureUnitState();

	tex->setAlphaOperation(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL,
			Ogre::LBS_CURRENT, alpha);

	if(alpha<1.0) {
		mMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
		mMaterial->setDepthWriteEnabled(false);
	} else {
		mMaterial->setSceneBlending(Ogre::SBT_REPLACE);
		mMaterial->setDepthWriteEnabled(true);
	}
}


// caller must make sure that data buffer is large enough
bool convertImg(const Img<>& img, uint8* data, double s_min, double s_max, const ContinuousColormap& colormap)
{
	size_t widthStep = img.step();
	int width  = img.width();
	int height = img.height();

	int imax = 65535; // default for 16 bit, 2^16
	if(s_max>=1.0f)
		imax = s_max;
	int imin = s_min;
	float imax_min = (imax-imin);

	if(s_max<0.0)
		s_max=1.0;

	switch (img.depth())
	{
	case CV_16U:
		if (img.channels() == 1) {
			uint8 *destPtr = data;
			for (int y = 0; y < height; y++)
			{
				const uint16_t *imagePtr = (const uint16_t *) img.data(y);
				for (int x = 0; x < width; x++) {
					int val = (*imagePtr++);
					float f = (float)(val-imin)/imax_min;
					if(f<0.0f) f=0.0f;
					else if(f>1.0f) f=1.0f;
					Color::RGB color = colormap.getf(f);
					*destPtr++ = color.b * 255.0f;
					*destPtr++ = color.g * 255.0f;
					*destPtr++ = color.r * 255.0f;
				}
			}
			return true;
		}

	case CV_16S:
		if (img.channels() == 1) {
			uint8 *destPtr = data;
			for (int y = 0; y < height; y++)
			{
				const int16_t *imagePtr = (const int16_t *) img.data(y);
				for (int x = 0; x < width; x++) {
					int val = (*imagePtr++);
					float f = (float)(val-imin)/imax_min + 0.5f;
					if(f<0.0f) f=0.0f;
					else if(f>1.0f) f=1.0f;
					Color::RGB color = colormap.getf(f);
					*destPtr++ = color.b * 255.0f;
					*destPtr++ = color.g * 255.0f;
					*destPtr++ = color.r * 255.0f;
				}
			}
			return true;
		}

	case CV_32F:
		if (img.channels() == 1) {
			uint8 *destPtr = data;
			for (int y = 0; y < height; y++)
			{
				const float *imagePtr = (const float *) img.data(y);
				for (int x = 0; x < width; x++) {
					float val = (*imagePtr++);
					float f = (val-s_min)/(s_max - s_min);
					if(f<0.0f) f=0.0f;
					else if(f>1.0f) f=1.0f;
					Color::RGB color = colormap.getf(f);
					*destPtr++ = color.b * 255.0f;
					*destPtr++ = color.g * 255.0f;
					*destPtr++ = color.r * 255.0f;
				}
			}
			return true;
		}

	case CV_64F:
		if (img.channels() == 1) {
			uint8 *destPtr = data;
			for (int y = 0; y < height; y++)
			{
				const double *imagePtr = (const double *) img.data(y);
				for (int x = 0; x < width; x++) {
					double val = (*imagePtr++);
					double f = (val-s_min)/(s_max - s_min);
					if(f<0.0) f=0.0;
					else if(f>1.0) f=1.0;
					Color::RGB color = colormap.getf(f);
					*destPtr++ = color.b * 255.0f;
					*destPtr++ = color.g * 255.0f;
					*destPtr++ = color.r * 255.0f;
				}
			}
			return true;
		}
	}

	return false;
}



void ImageObject::setImage(const Img<>& img, double smin, double smax, const ContinuousColormap& colormap, bool enableMipmapping)
{
	// do not try to create an empty texture, to prevent Ogre exceptions on some graphics hardware
	if(img.isEmpty())
		return;

	boost::scoped_array<uint8> data;
	Ogre::PixelFormat pxf;
	if(img.channels()==1 && (img.depth()==CV_8U || img.depth()==CV_8S))
		pxf = Ogre::PF_L8;
	else if(img.channels()==3 && img.depth()==CV_8U)
		pxf = Ogre::PF_R8G8B8;
	else if(img.channels()==4 && img.depth()==CV_8U)
		pxf = Ogre::PF_X8R8G8B8;
	else { // need to convert
		data.reset(new uint8[img.width()*img.height()*3]);
		pxf = Ogre::PF_R8G8B8;
		if(!convertImg(img, data.get(), smin, smax, colormap))
			MIRA_THROW(XInvalidParameter, "Unsupported image type");
	}

	if (!mTexture.isNull())
	{
		// delete texture if size or pixel format does not match
		if ( mTexture->getSrcWidth() != (size_t)img.width() ||
			 mTexture->getSrcHeight() != (size_t)img.height() ||
			 mTexturePxf != pxf ||
			 (bool(mTexture->getNumMipmaps()>0) != enableMipmapping))
		{
			Ogre::TextureManager::getSingleton().remove(mTexture->getName());
			mTexture.setNull();
		}
	}
	// create new texture if not exists
	if (mTexture.isNull())
	{
		static int counter = 0;
		mTexture = Ogre::TextureManager::getSingleton().createManual(
				"ImageObject"+toString(this)+"/Texture"+toString(counter++),
				"MIRAMaterials", Ogre::TEX_TYPE_2D, img.width(), img.height(),
				enableMipmapping?Ogre::MIP_DEFAULT:0, pxf ,
				Ogre::TU_DYNAMIC_WRITE_ONLY);
		// set texture to material
		Ogre::TextureUnitState* tex = getTextureState();
		tex->setTextureName(mTexture->getName());
		tex->setTextureFiltering(Ogre::FO_NONE,Ogre::FO_NONE,
		                    enableMipmapping?Ogre::FO_LINEAR:Ogre::FO_NONE);
	}

	// copy data from image
	const uint8* d = img.data();
	if(data)
		d = data.get();
	Ogre::PixelBox npb(img.width(), img.height(), 1, pxf, const_cast<uint8*>(d));
	mTexture->getBuffer(0, 0)->blitFromMemory(npb);

	// generate mipmaps
	if(mTexture->getNumMipmaps()>0) {
		// create temp memory buffer that is large enough to hold the largest mipmap
		boost::scoped_array<uint8> data(new uint8[img.width()*img.height()*Ogre::PixelUtil::getNumElemBytes(pxf)]);
		for(size_t i=1; i<mTexture->getNumMipmaps(); ++i)
		{
			Ogre::HardwarePixelBufferSharedPtr mipmap = mTexture->getBuffer(0, i);
			Ogre::PixelBox mpb(mipmap->getWidth(), mipmap->getHeight(), 1, pxf, data.get());
			Ogre::Image::scale(npb,mpb,Ogre::Image::FILTER_BICUBIC);
			mipmap->blitFromMemory(mpb);
		}
	}

	mTexturePxf = pxf;
}

void ImageObject::setImage(const Img<>& img, bool enableMipmapping)
{
	static GrayscaleColormap graymap;
	setImage(img, 0.0, -1.0, graymap,enableMipmapping);
}


Ogre::TextureUnitState* ImageObject::getTextureState()
{
	Ogre::Pass* pass = mMaterial->getTechnique(0)->getPass(0);
	Ogre::TextureUnitState* tex = NULL;
	if (pass->getNumTextureUnitStates() > 0)
		tex = pass->getTextureUnitState(0);
	else
		tex = pass->createTextureUnitState();
	return tex;
}

//////////////////////////////////////////////////////////////////////////////

}
