/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CameraOrbitTool.C
 *    Implementation of CameraOrbitTool2D.
 *
 * @author Tim Langner
 * @date   2013/07/09
 */

#include <visualization/2d/CameraOrbitTool.h>

#include <QTimeLine>
#include <QGraphicsView>

namespace mira {

///////////////////////////////////////////////////////////////////////////////


CameraOrbitTool2D::CameraOrbitTool2D()
{
	mPan = false;
	mRotate = false;
}

QIcon CameraOrbitTool2D::getIcon()
{
	QPixmap pixmap(":/icons/ToolCameraOrbit.png");
	return QIcon(pixmap);
}

void CameraOrbitTool2D::onMousePressed(QMouseEvent* e)
{
	updateOwnRepresentation();
	if (e->button() == Qt::LeftButton)
	{
		mPan = true;
		getSite()->getViewManager()->setCursor(Qt::ClosedHandCursor);
	}
	else if (e->button() == Qt::RightButton)
	{
		mRotate = true;
		getSite()->getViewManager()->setCursor(Qt::ClosedHandCursor);
	}
	mOldMousePos  = e->pos();
	mMouseDownPos = e->pos();
}

void CameraOrbitTool2D::onMouseReleased(QMouseEvent* e)
{
	if (e->button() == Qt::LeftButton)
		mPan = false;

	if (e->button() == Qt::RightButton)
		mRotate = false;

	if (!mPan && !mRotate)
		getSite()->getViewManager()->setCursor(Qt::ArrowCursor);
}

void CameraOrbitTool2D::onMouseMoved(QMouseEvent* e)
{
	QPoint delta = e->pos() - mOldMousePos;
	if (e->buttons() == (Qt::LeftButton | Qt::RightButton) ||
		e->buttons() == Qt::MidButton) {
		zoom(delta.y(),mMouseDownPos);
	} else if (mPan) {
		mCamera.position.x() += (delta.x() *  cos(mCamera.orientation.rad()) + delta.y() * sin(mCamera.orientation.rad()))/mCamera.scale;

		// As the Y axis of the our view is flipped we flip Y by inverting the sign here as well.
		mCamera.position.y() -= (delta.x() * -sin(mCamera.orientation.rad()) + delta.y() * cos(mCamera.orientation.rad()))/mCamera.scale;
		getSite()->setCamera(mCamera);
	} else if (mRotate) {
		rotate(delta.x(),mMouseDownPos);
	}
	mOldMousePos  = e->pos();
}

void CameraOrbitTool2D::onMouseWheel(QWheelEvent * e)
{
	updateOwnRepresentation();
	int numDegrees = e->delta() / 8;
	int numSteps = numDegrees / 15;  // see QWheelEvent documentation
	zoom(numSteps*10,e->pos());
}

void CameraOrbitTool2D::zoom(int numSteps, const QPoint& pivot)
{
	if(numSteps==0)
		return;

	float factor;
	if(numSteps>=0)
		factor = 1.0 + numSteps/200.0f;
	else
		factor = 1.0f / (1.0f+(-numSteps/200.0f));

	if(mCamera.scale<0.01f && factor<1.0f)
		return; // limit scale

	// Apply the scale and see how our pivot point moved
	auto before = getSite()->getViewManager()->mapToScene(pivot);
	mCamera.scale *= factor;
	getSite()->setCamera(mCamera);
	auto after = getSite()->getViewManager()->mapToScene(pivot);

	// Correct camera position by offset that our pivot point moved
	mCamera.position(0) += (after.x() - before.x());
	mCamera.position(1) += (after.y() - before.y());
	getSite()->setCamera(mCamera);
}

void CameraOrbitTool2D::rotate(int numSteps, const QPoint& pivot)
{
	if(numSteps==0)
		return;

	// Apply the scale and see how our pivot point moved
	auto before = getSite()->getViewManager()->mapToScene(pivot);
	mCamera.orientation += numSteps;
	getSite()->setCamera(mCamera);
	auto after = getSite()->getViewManager()->mapToScene(pivot);

	// Correct camera position by offset that our pivot point moved
	mCamera.position(0) += (after.x() - before.x());
	mCamera.position(1) += (after.y() - before.y());
	getSite()->setCamera(mCamera);
}

int CameraOrbitTool2D::getDisplaceShortCut() const
{
	return Qt::Key_Control;
}

void CameraOrbitTool2D::activate()
{
	updateOwnRepresentation();
	getSite()->getViewManager()->setInteractive(false);
}

void CameraOrbitTool2D::deactivate()
{
	getSite()->getViewManager()->setInteractive(true);
	mPan = false;
	mRotate = false;
}

QGraphicsItem* CameraOrbitTool2D::setupScene(QGraphicsScene* mgr)
{
	return NULL;
}

void CameraOrbitTool2D::updateOwnRepresentation()
{
	if(getSite()==NULL)
		return;
	mCamera = getSite()->getCamera();
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::CameraOrbitTool2D, mira::VisualizationTool2D)
