/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization3DBasic.h
 *    A base class for simple 3D visualizations.
 *
 * @author Erik Einhorn
 * @date   2011/06/01
 */

#ifndef _MIRA_VISUALIZATION3DBASIC_H_
#define _MIRA_VISUALIZATION3DBASIC_H_

#ifndef Q_MOC_RUN
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>
#endif

#include <visualization/Visualization3D.h>

#include <widgets/OgreUtils.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Inherit from this class if you want to implement a
 * simple 3D visualization.
 * The template parameter specifies what type of data your
 * class visualizes.
 * The class already creates a scene node for you and updates the
 * transformation of the node according to the frame id of the data.
 * All you need to implement is the setupScene method.
 * For more complex visualizations please use the Visualization3D
 * directly.
 */
template <typename T>
class Visualization3DBasic : public Visualization3D
{
public:

	/**
	 * Constructor taking a name of the visualized channel
	 */
	Visualization3DBasic(const std::string channelDisplayName = "Channel") :
		mChannelDisplayName(channelDisplayName),
		mNode(NULL) {
		mDataChannel.setDataChangedCallback(
				boost::bind(&Visualization3DBasic::dataChangedCallback, this, _1));
		mDataChannel.setChannelChangedCallback(
				boost::bind(&Visualization3DBasic::channelChangedCallback, this));
	}

	virtual ~Visualization3DBasic() {
		if(getSite()==NULL)
			return;
		getSite()->getSceneManager()->destroySceneNode(mNode);
	}

	template <typename Reflector>
	void reflect(Reflector& r) {
		Visualization3D::reflect(r);
		channelProperty(r, mChannelDisplayName.c_str(), mDataChannel,
		                "The channel to be visualized");
	}

public:

	virtual void setupScene(IVisualization3DSite* site) {
		Ogre::SceneManager* mgr = site->getSceneManager();
		mNode = mgr->getRootSceneNode()->createChildSceneNode();
		setupScene(mgr, mNode);
		mNode->setVisible(isEnabled());
	}

	/**
	 * Implement this method to setup the scene.
	 * The node is already created by this class and is a child node
	 * of the root scene node.
	 */
	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node) = 0;

	virtual Ogre::SceneNode* getNode()
	{
		return mNode;
	}

public:

	void setEnabled(bool enabled) {
		Visualization3D::setEnabled(enabled);
		if (mNode != NULL)
			mNode->setVisible(enabled);
	}

public:

	virtual DataConnection getDataConnection() {
		return DataConnection(mDataChannel);
	}

protected:

	virtual void update(Duration dt) {
		if (!mDataChannel.isValid() || mDataChannel.getDataUpdateCount() == 0)
			return;

		if (mDataFrameID.empty())
			MIRA_THROW(TransformerBase::XTransform,
			           "Data has no frame for transforming it into the view's camera frame");

		RigidTransform3f d;
		try {
			d = getAuthority().template getTransform<RigidTransform3f>(
				mDataFrameID, mDataTimestamp,
				getSite()->getCameraFrame(), Time::now(),
				getSite()->getFixedFrame(), LinearInterpolatorNearestNeighbourExtrapolator());
		}
		catch(Exception& ex) {	// fall back to non interpolating versions
			d = getAuthority().template getTransform<RigidTransform3f>(
				mDataFrameID, mDataTimestamp,
				getSite()->getCameraFrame(), Time::now(),
				getSite()->getFixedFrame());
		}

		OgreUtils::setTransform(mNode, d);
	}

	void dataChangedCallback(ChannelRead<T> data) {
		mDataFrameID = data->frameID;
		mDataTimestamp = data->timestamp;
		dataChanged(data);
	}

	void channelChangedCallback() {
		mDataFrameID = "";
		try {
			this->setName(mDataChannel.getID());
		}
		catch (XRuntime&) {}
		channelChanged();
	}

	/**
	 * Overwrite this method to get notified when the connected channel changes.
	 */
	virtual void channelChanged() {}

	/**
	 * Overwrite this method to listen on channel callbacks.
	 */
	virtual void dataChanged(ChannelRead<T> data) {
		onDataChanged(data);	// only for backward compatibility
	}

	/**
	 * \deprecated. Only for backward compatibility
	 */
	virtual void onDataChanged(ChannelRead<T> data) {}

protected:

	ChannelProperty<T> mDataChannel;
	std::string mChannelDisplayName;

	std::string mDataFrameID;
	Time mDataTimestamp;

	Ogre::SceneNode* mNode;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
