/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeTest.C
 *    Test cases for tape.
 *
 * @author Tim Langner
 * @date   2011/01/20
 */

#include <boost/test/unit_test.hpp>

#include <fw/TapeRecorder.h>
#include <fw/TapePlayer.h>
#include <transform/RigidTransform.h>
#include <utils/Stamped.h>

using namespace mira;

template <typename T>
void readMessage(Tape* tape, const Tape::MessageIndex& index, Stamped<T>& oValue)
{
	Buffer<uint8> buffer;
	Duration time;
	tape->readMessage(index, oValue.frameID, oValue.sequenceID, buffer, time);
	oValue.timestamp = tape->getStartTime() + time;

	BinaryBufferDeserializer d(&buffer);
	d.deserialize(oValue.internalValueRep(), false);
}

BOOST_AUTO_TEST_CASE(ReadWriteTapeTest)
{
	Time now = Time::now();
	{
		Tape t;
		t.open("test.tape", Tape::WRITE);
		Stamped<RigidTransform2f> v(RigidTransform2f(10.0f, 5.0f, pi<float>()), now, "MyFrame");
		t.write("MyChannel", v);
		t.close();
	}
	{
		Tape t;
		t.open("test.tape", Tape::READ);
		const Tape::ChannelMap& channels = t.getChannels();
		BOOST_CHECK_EQUAL(channels.size(), 1);
		BOOST_CHECK_EQUAL(channels.begin()->first, "MyChannel");
		BOOST_CHECK_EQUAL(channels.begin()->second.messages.size(), 1);
		Stamped<RigidTransform2f> v;
		readMessage(&t, t.getChannels().begin()->second.messages[0], v);
		BOOST_CHECK_CLOSE(v.x(), 10.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.y(), 5.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.phi(), pi<float>(), 0.0001f);
		BOOST_CHECK_EQUAL(v.timestamp, now);
		BOOST_CHECK_EQUAL(v.frameID, "MyFrame");
	}
}

BOOST_AUTO_TEST_CASE(ReadWriteCompressedTapeTest)
{
	Time now = Time::now();
	{
		Tape t;
		t.open("compressed.tape", Tape::WRITE);
		Stamped<RigidTransform2f> v(RigidTransform2f(10.0f, 5.0f, pi<float>()), now, "MyFrame");
		t.write("MyChannel", v, true);
		t.close();
	}
	{
		Tape t;
		t.open("compressed.tape", Tape::READ);
		const Tape::ChannelMap& channels = t.getChannels();
		BOOST_CHECK_EQUAL(channels.size(), 1);
		BOOST_CHECK_EQUAL(channels.begin()->first, "MyChannel");
		BOOST_CHECK_EQUAL(channels.begin()->second.messages.size(), 1);
		Stamped<RigidTransform2f> v;
		readMessage(&t, t.getChannels().begin()->second.messages[0], v);
		BOOST_CHECK_CLOSE(v.x(), 10.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.y(), 5.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.phi(), pi<float>(), 0.0001f);
		BOOST_CHECK_EQUAL(v.timestamp, now);
		BOOST_CHECK_EQUAL(v.frameID, "MyFrame");
	}
}

BOOST_AUTO_TEST_CASE(TapeRecorderTest)
{
	{
		const char* argv[] = {"TapeRecorderTest", "-d0",
							  "--config-string",
							  "<root>"
							  "<unit id=\"TapePlayer\" class=\"mira::TapePlayerUnit\">"
								  "<AutoPlayFile><item>test.tape</item></AutoPlayFile>"
								  "<Loop>true</Loop>"
								  "<LoopCount>-1</LoopCount>"
							  "</unit>"
							  "</root>"};
		Framework fw(4,(char**)argv);
		fw.load();
		fw.start();

		TapeRecorder recorder;
		recorder.addChannel("/MyChannel");

		recorder.record("record1.tape");
		MIRA_SLEEP(100);

		recorder.record("record2.tape");
		MIRA_SLEEP(100);
	}

	{
		Tape t;
		t.open("record1.tape", Tape::READ);
		const Tape::ChannelMap& channels = t.getChannels();
		BOOST_CHECK_EQUAL(channels.size(), 1);
		BOOST_CHECK_EQUAL(channels.begin()->first, "/MyChannel");
		BOOST_CHECK_EQUAL(channels.begin()->second.messages.size(), 1);
		Stamped<RigidTransform2f> v;
		readMessage(&t, t.getChannels().begin()->second.messages[0], v);
		BOOST_CHECK_CLOSE(v.x(), 10.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.y(), 5.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.phi(), pi<float>(), 0.0001f);
		BOOST_CHECK_EQUAL(v.frameID, "MyFrame");
	}

	{
		Tape t;
		t.open("record2.tape", Tape::READ);
		const Tape::ChannelMap& channels = t.getChannels();
		BOOST_CHECK_EQUAL(channels.size(), 1);
		BOOST_CHECK_EQUAL(channels.begin()->first, "/MyChannel");
		BOOST_CHECK_EQUAL(channels.begin()->second.messages.size(), 1);
		Stamped<RigidTransform2f> v;
		readMessage(&t, t.getChannels().begin()->second.messages[0], v);
		BOOST_CHECK_CLOSE(v.x(), 10.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.y(), 5.0f, 0.0001f);
		BOOST_CHECK_CLOSE(v.phi(), pi<float>(), 0.0001f);
		BOOST_CHECK_EQUAL(v.frameID, "MyFrame");
	}
}
