/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkLoaderTest.C
 *    Test cases for framework loaders.
 *
 * @author Tim Langner
 * @date   2012/04/07
 */

#include <boost/test/unit_test.hpp>

#include <transform/Pose.h>
#include <fw/Framework.h>
#include <fw/UnitManager.h>

using namespace mira;

BOOST_AUTO_TEST_CASE(FrameworkLoaderTest)
{
	const char* argv[] = {"FrameworkLoaderTest",
	                      "${find FrameworkLoaderTest.xml}",
	                      "-d1",
	                      "--no-colors"};
	Framework fw(4,(char**)argv);
	fw.load();
	fw.start();
	BOOST_CHECK(fw.getUnitManager()->getUnit("/IntPublisher") != NULL);
	BOOST_CHECK(fw.getUnitManager()->getUnit("/ns1/IntSubscriber") != NULL);
	BOOST_CHECK(fw.getUnitManager()->getUnit("/ns2/IntSubscriber") == NULL);

	fw.getChannelManager().getConcreteChannel<int>("/Int");
	BOOST_CHECK_THROW(fw.getChannelManager().getConcreteChannel<float>("/Float"), XUnknownChannel);

	FrameworkTransformer::NodePtr rootNode = fw.getTransformer()->getNode("/RootFrame");
	FrameworkTransformer::NodePtr pose2Node = fw.getTransformer()->getNode("/Pose2Frame");
	FrameworkTransformer::NodePtr poseCov2Node = fw.getTransformer()->getNode("/PoseCov2Frame");
	FrameworkTransformer::NodePtr pose3Node = fw.getTransformer()->getNode("/Pose3Frame");
//	FrameworkTransformer::NodePtr poseCov3Node = fw.getTransformer()->getNode("/PoseCov3Frame");
	BOOST_CHECK(rootNode != NULL);
	BOOST_CHECK(pose2Node != NULL);
	BOOST_CHECK(poseCov2Node != NULL);
	BOOST_CHECK(pose3Node != NULL);
	//BOOST_CHECK(poseCov3Node != NULL);

	Pose2 p2 = fw.getTransformer()->getTransform<Pose2>(pose2Node, rootNode, Time::now());
	BOOST_CHECK_CLOSE(p2.x(), 10.0f, 0.001f);
	BOOST_CHECK_CLOSE(p2.y(), 5.0f, 0.001f);
	BOOST_CHECK_CLOSE(p2.phi(), deg2rad(90.0f), 0.001f);

	PoseCov2 pc2 = fw.getTransformer()->getTransform<PoseCov2>(poseCov2Node, rootNode, Time::now());
	BOOST_CHECK_CLOSE(pc2.x(), 10.0f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.y(), 5.0f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.phi(), deg2rad(90.0f), 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(0,0), 0.1f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(1,1), 0.2f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(2,2), 0.3f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(1,0), 0.0f, 0.001f);

	Pose3 p3 = fw.getTransformer()->getTransform<Pose3>(pose3Node, rootNode, Time::now());
	BOOST_CHECK_CLOSE(p3.x(), 10.0f, 0.001f);
	BOOST_CHECK_CLOSE(p3.y(), 5.0f, 0.001f);
	BOOST_CHECK_CLOSE(p3.z(), 2.5f, 0.001f);
	BOOST_CHECK_CLOSE(p3.yaw(), deg2rad(0.0f), 0.001f);
	BOOST_CHECK_CLOSE(p3.pitch(), deg2rad(90.0f), 0.001f);
	BOOST_CHECK_CLOSE(p3.roll(), deg2rad(10.0f), 0.001f);
}
