/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Status.C
 *    Implementation of Status.h.
 *
 * @author Tim Langner
 * @date   2011/02/03
 */

#include <fw/Status.h>

#include <fw/Framework.h>
#include <fw/ErrorService.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void DiagnosticsModule::bootup(const std::string& message,
                               const std::string& trText)
{
	Status s(Status::BOOTUP, "Booting up", trText, message);
	// Test if that status is already set
	if (mBootUpStatus && *mBootUpStatus == s)
		return;

	MIRA_LOG(NOTICE) << mName << " Booting up: " << trText << ": " << message;
	mBootUpStatus.reset(s);
}

void DiagnosticsModule::bootupFinished()
{
	MIRA_LOG(NOTICE) << mName << " Bootup finished";
	mBootUpStatus.reset();
}

void DiagnosticsModule::recoverFinished()
{
	MIRA_LOG(NOTICE) << mName << " Recover finished";
	mRecoverStatus.reset();
}

void DiagnosticsModule::recover(const std::string& message,
                                const std::string& trText)
{
	Status s(Status::RECOVER, "Recovering", trText, message);
	// Test if that status is already set
	if (mRecoverStatus && *mRecoverStatus == s)
		return;

	MIRA_LOG(ERROR) << mName << " Recovering: " << trText << ": " << message;
	mRecoverStatus.reset(s);
}

void DiagnosticsModule::ok(const std::string& category)
{
	if (mStatusMap.empty())
		return;
	if (!category.empty())
		// erase all errors and warnings in that category
		mStatusMap.erase(category);
	else
		// erase all errors and warnings
		mStatusMap.clear();
}

bool DiagnosticsModule::warning(const std::string& category,
                                const std::string& message,
                                const std::string& trText)
{
	if(setStatus(Status::WARNING, category, message, trText)) {
		MIRA_LOG(WARNING) << mName << " "<< trText << ": " << message;
		return true;
	}
	return false;
}

bool DiagnosticsModule::error(const std::string& category,
                              const std::string& message,
                              const std::string& trText)
{
	if(setStatus(Status::ERROR, category, message, trText)) {
		MIRA_LOG(ERROR) << mName << " "<< trText << ": " << message;
		return true;
	}
	return false;
}

bool DiagnosticsModule::setStatus(Status::StatusMode mode,
                                  const std::string& category,
                                  const std::string& message,
                                  const std::string& trText)
{
	Status s(mode, category, trText, message);
	// Test if that status is already set
	StatusMap::iterator i = mStatusMap.find(category);
	if (i != mStatusMap.end() && i->second == s)
		return false;
	mStatusMap[category] = s;
	return true;
}

Status::StatusMode DiagnosticsModule::getStatus() const
{
	Status::StatusMode mode = Status::OK;
	if (mBootUpStatus)
		mode = Status::BOOTUP;
	if (mRecoverStatus)
		mode = Status::RECOVER;
	foreach(const auto& s, mStatusMap)
	{
		if (s.second.mode == Status::ERROR)
			return Status::ERROR;
		if (s.second.mode == Status::WARNING)
			mode = Status::WARNING;
	}
	if (mode != Status::OK)
		return mode;

	if (hasHeartbeatTimeout())
		return Status::ERROR;

	return Status::OK;
}

DiagnosticsModule::StatusMap DiagnosticsModule::getStatusMap() const
{
	StatusMap r = mStatusMap;
	if (hasHeartbeatTimeout())
		r["Heartbeat"] = Status(Status::ERROR, "HeartBeat",
		                        "Not working - watchdog timed out",
		                        MakeString() << "Last heartbeat "
		                        << Time::now()-*mLastHeartbeat << " ago");
	
	if (mBootUpStatus)
		r["Booting up"] = *mBootUpStatus;
	if (mRecoverStatus)
		r["Recover"] = *mRecoverStatus;
	return r;
}

///////////////////////////////////////////////////////////////////////////////

void StatusManager::registerDiagnostics(const std::string& name,
                                        DiagnosticsModulePtr ptr)
{
	boost::mutex::scoped_lock lock(mModuleMutex);
	mModules[name] = ptr;
	mModuleOrder.push_back(name);
}

void StatusManager::unregisterDiagnostics(const std::string& name)
{
	boost::mutex::scoped_lock lock(mModuleMutex);
	for(auto i=mModuleOrder.begin(); i!=mModuleOrder.end(); ++i)
		if (*i == name)
		{
			mModuleOrder.erase(i);
			break;
		}
	mModules.erase(name);
}

void StatusManager::unregisterDiagnostics(DiagnosticsModulePtr ptr)
{
	foreach(const auto& m, mModules)
		if (m.second == ptr)
		{
			unregisterDiagnostics(m.first);
			return;
		}
}

std::vector<std::string> StatusManager::getDiagnosticModules() const
{
	return mModuleOrder;
}

Status::StatusMode StatusManager::getStatus() const
{
	boost::mutex::scoped_lock lock(mModuleMutex);
	// get the "most severe" mode from all modules
	Status::StatusMode mostSevereMode = Status::OK;
	foreach(const auto& s, mModules)
	{
		Status::StatusMode moduleMode = s.second->getStatus();
		if (mostSevereMode < moduleMode)
			mostSevereMode = moduleMode;
	}
	return mostSevereMode;
}

Status::StatusMode StatusManager::getStatus(const std::string& diagnosticModule) const
{
	boost::mutex::scoped_lock lock(mModuleMutex);
	auto iter = mModules.find(diagnosticModule);
	if (iter == mModules.end())
		return Status::OK;
	return iter->second->getStatus();
}

StatusManager::StatusMap StatusManager::getStatusMap() const
{
	boost::mutex::scoped_lock lock(mModuleMutex);
	StatusMap r;
	foreach(const auto& m, mModules)
	{
		DiagnosticsModule::StatusMap smap = m.second->getStatusMap();
		foreach(const auto& s, smap)
			r.insert(std::make_pair(m.first, s.second)); 
	}
	return r;
}

DiagnosticsModule::StatusMap StatusManager::getStatusMap(const std::string& diagnosticModule) const
{
	boost::mutex::scoped_lock lock(mModuleMutex);
	auto iter = mModules.find(diagnosticModule);
	if (iter == mModules.end())
		return DiagnosticsModule::StatusMap();
	return iter->second->getStatusMap();
}

///////////////////////////////////////////////////////////////////////////////

}
