/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelReadWrite.h
 *    Classes for automatic locking/unlocking when reading and writing to channels.
 *    @note You should never include this file directly. Include fw/Channel.h instead!
 *
 * @author Erik Einhorn
 * @date   2010/09/08
 */

#ifndef _MIRA_CHANNELLOCKEDREADWRITE_H_
#define _MIRA_CHANNELLOCKEDREADWRITE_H_

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#include <boost/thread/shared_mutex.hpp>
#include <boost/thread/locks.hpp>
#endif

#include <fw/AbstractChannel.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward declaration
template<typename T>
class ConcreteChannel;

template<typename T>
class Channel;

// forward declaration
template <typename ConcreteChannelReadWrite>
struct ChannelReadWriteTraits {};

///////////////////////////////////////////////////////////////////////////////

/**
 * Internally used by ChannelReadWriteBase!
 * It contains the information about the channel and slot that can be
 * shared between different ChannelRead/Write objects.
 * In the destructor, this class calls the appropriate finish-method
 * of the concrete ChannelRead/Write container below.
 */
template<typename DerivedContainer>
class ChannelReadWriteShared
{
public:
	typedef typename ChannelReadWriteTraits<DerivedContainer>::ChannelType ChannelType;
	typedef typename ChannelReadWriteTraits<DerivedContainer>::Lock    Lock;
	typedef typename ChannelReadWriteTraits<DerivedContainer>::SlotType Slot;
	typedef typename ChannelReadWriteTraits<DerivedContainer>::ValueType ValueType;

	typedef ChannelType*               ChannelTypePtr;


public:

	ChannelReadWriteShared(ChannelTypePtr iChannel, Slot* iSlot) :
		channel(iChannel), slot(iSlot), lock(iSlot->lock, boost::adopt_lock) {}

	~ChannelReadWriteShared() {
		// if our destructor was invoked due to stack unwinding in a exception
		// handling machanism, we do NOT write the data but discard it. The
		// data is written only, if the destructor is called as we got out
		// of scope WITHOUT a raised exception.
		if(!std::uncaught_exception())
			DerivedContainer::finish(this); // may throw
		else
			DerivedContainer::discard(this); // must not throw
	}

public:

	/// pointer to the channel our data/slot belongs to
	ChannelTypePtr channel;

	/**
	 * the slot in that channel we are pointing on (this pointer is valid
	 * unless the channel's buffer is destructed or promoted!)
	 */
	Slot*      slot;

	/// a shared lock for that data in the channel's buffer
	Lock       lock;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @note: NEVER EXPOSE THE CHANNELPTR OR CHANNEL DIRECTLY TO THE USER.
 *        The user is allowed to access the channel using the Channel<T>
 *        proxy ONLY!
 */
template<typename Derived>
class ChannelReadWriteCommonBase
{
public:

	typedef ChannelReadWriteShared<Derived> Shared;
	typedef boost::shared_ptr<Shared>       SharedPtr;

	// import some important types from our Shared class
	typedef typename Shared::ChannelType    ChannelType;
	typedef typename Shared::ChannelTypePtr ChannelTypePtr;
	typedef typename Shared::Slot           Slot;
	typedef typename Shared::ValueType      ValueType;

public:

	/// Constructs an empty (invalid) ChannelReadWriteBase object.
	ChannelReadWriteCommonBase() {}

	/// Constructs a valid ChannelReadWriteBase object that is assigned to a channel and slot.
	ChannelReadWriteCommonBase(ChannelTypePtr channel, Slot* slot) :
		shared(new Shared(channel, slot) )
	{}

public:

	/**
	 * Returns true, if data was assigned to the ChannelRead or ChannelWrite
	 * and if this data is locked. If this method returns true, you can
	 * safely access the underlying data.
	 *
	 * If this method returns false, no data was assigned to the ChannelRead /
	 * ChannelWrite or the lock on the data was lost. Hence, it is not safe
	 * to access the underlying data. If you try to access the data anyway
	 * you will get an XRuntime exception.
	 */
	bool isValid() const {
		// channel must be assigned and data must be locked
		return shared.get()!=NULL && shared->lock.owns_lock();
	}

public:

	const std::string& getChannelID() {
		static std::string emptyString;
		return shared.get()!=NULL ? shared->channel->getID() : emptyString;
	}

	std::string getTypename() const {
		return shared.get()!=NULL ? shared->channel->getTypename() : "";
	}

	TypeMetaPtr getTypeMeta() const {
		return shared.get()!=NULL ? shared->channel->getTypeMeta() : TypeMetaPtr();
	}

	const Time& getTimestamp() const {
		checkValid();
		return shared->slot->timestamp();
	}

	/**
	 *  Returns the number of shared instances for this data,
	 *  may be useful for debugging purposes.
	 */
	int getUseCount() const {
		return shared.get()!=NULL ? shared.use_count() : 0;
	}

public:

	/**
	 * For internal use only.
	 * Used by remote components of the framework and by player and recorder.
	 * Returns current flags of this slot;
	 */
	uint32 getFlags() const {
		checkValid();
		return shared->slot->flags;
	}

	/**
	 * For internal use only.
	 * Used by remote components of the framework and by player and recorder.
	 * Add additional flags to the slots flags.
	 */
	void addFlags(uint32 moreFlags) const {
		checkValid();
		shared->slot->flags |= moreFlags;
	}

protected:

	/// checks if we are still locked, if not it throws a XAccessViolation
	void checkValid() const {
		if(shared.get()==NULL)
			MIRA_THROW(XAccessViolation, "Trying to access ChannelRead / "
			           "ChannelWrite that was not assigned with valid data");

		if(!shared->lock.owns_lock())
			MIRA_THROW(XAccessViolation, "Trying to access data of ChannelRead "
			           "/ ChannelWrite after losing the lock");
	}

protected:
	// these read and write methods are protected by default. They will be
	// made public in the final ChannelRead/Write classes.

	const Buffer<uint8>& readSerializedValue() {
		checkValid();
		const Buffer<uint8>& res =
				shared->channel->getBuffer()->readSerializedValue(shared->slot);

		// ??? can we ever have an empty vector here ???
		// I think we can't, since:
		//    1. we either have a typed buffer where we can deserialize the
		//       data, if it doesn't exist.
		//    2. If we have an untyped buffer, someone has probably written
		//       some serialized data, otherwise the reader would never get a
		//       ChannelRead object since the channel is empty. He would have
		//       got an exception when trying to obtain the read object.
		assert(!res.empty());
		return res;
	}

	/// Same as above, but allows to specify codecs for serialization.
	Buffer<uint8> readSerializedValue(std::list<BinarySerializerCodecPtr>& codecs) {
		checkValid();
		return shared->channel->getBuffer()->readSerializedValue(shared->slot, codecs);
	}

	void readJSON(JSONValue& oValue) {
		checkValid();
		shared->channel->getBuffer()->readJSON(shared->slot, oValue);
	}

protected:

	void writeSerializedValue(Buffer<uint8> data) {
		checkValid();
		shared->channel->getBuffer()->writeSerializedValue(shared->slot, std::move(data));
	}

	void writeJSON(const JSONValue& value) {
		checkValid();
		shared->channel->getBuffer()->writeJSON(shared->slot, value);
	}


protected:
	SharedPtr shared;
};



template<typename Derived, typename T>
class ChannelReadWriteBase : public ChannelReadWriteCommonBase<Derived>
{
public:

	typedef ChannelReadWriteCommonBase<Derived> Base;

	// push down some important types from our base
	typedef typename Base::Shared     Shared;
	typedef typename Base::ChannelTypePtr ChannelTypePtr;
	typedef typename Base::Slot       Slot;
	typedef typename Base::ValueType  ValueType;

public:

	/// Constructs an empty (invalid) ChannelReadWriteBase object.
	ChannelReadWriteBase() {}

	/// Constructs a valid ChannelReadWriteBase object that is assigned to a channel and slot.
	ChannelReadWriteBase(ChannelTypePtr channel, Slot* slot) : Base(channel,slot) {}

protected:

	/**
	 * Returns a reference on the data.
	 * @throw Throws XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method
	 */
	ValueType& internalValue() {
		this->checkValid();
		return this->shared->slot->data;
	}

	/**
	 * Returns a const reference on the data.
	 * @throw Throws XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method
	 */
	const ValueType& internalValue() const {
		this->checkValid();
		return this->shared->slot->data;
	}

};


/**
 * Specialization for untyped channels which just return the
 * StampedHeader as data.
 */
template<typename Derived>
class ChannelReadWriteBase<Derived, void> : public ChannelReadWriteCommonBase<Derived>
{
public:

	typedef ChannelReadWriteCommonBase<Derived> Base;

	// push down some important types from our base
	typedef typename Base::Shared     Shared;
	typedef typename Base::ChannelTypePtr ChannelTypePtr;
	typedef typename Base::Slot       Slot;
	typedef typename Base::ValueType  ValueType;

public:

	/// Constructs an empty (invalid) ChannelReadWriteBase object.
	ChannelReadWriteBase() {}

	/// Constructs a valid ChannelReadWriteBase object that is assigned to a channel and slot.
	ChannelReadWriteBase(ChannelTypePtr channel, Slot* slot) : Base(channel,slot) {}

protected:

	/**
	 * Returns a reference on the data.
	 * @throw Throws XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method
	 */
	ValueType& internalValue() {
		this->checkValid();
		return this->shared->channel->getBuffer()->getStampedHeader(this->shared->slot);
	}

	/**
	 * Returns a const reference on the data.
	 * @throw Throws XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method
	 */
	const ValueType& internalValue() const {
		this->checkValid();
		return this->shared->channel->getBuffer()->getStampedHeader(this->shared->slot);
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * An object that allows read access to data of a channel.
 * Use @ref Channel::read to obtain access to data of a channel.
 *
 * The ChannelRead automatically take care that the data is unlocked if the
 * ChannelRead object is no longer used and went out of scope. To
 * explicitly release the data by calling the @ref finish() method.
 *
 * You can freely copy and reassign the ChannelRead object to other
 * ChannelRead variables. The ChannelRead contains an internal reference
 * counting mechanism. The lock is freed only after all copies have
 * been destroyed or one copy called the @ref finish() method.
 */
template<typename T>
class ChannelRead : public ChannelReadWriteBase<ChannelRead<T>,T>
{
protected:

	typedef ChannelReadWriteBase<ChannelRead<T>,T> Base;

	// push down some important types from our base
	typedef typename Base::Shared     Shared;
	typedef typename Base::ChannelTypePtr ChannelTypePtr;
	typedef typename Base::Slot       Slot;
	typedef typename Base::ValueType  ValueType;

public:

	/**
	 * Default constructor that constructs a ChannelRead object that
	 * is invalid at the beginning and contains no data. Before you
	 * can use the object you need to assign a valid ChannelWrite
	 * object that was obtained using @ref Channel::read().
	 *
	 * You can use the isValid() method to check if a ChannelRead object
	 * is valid and contains locked data.
	 */
	ChannelRead() {}

	/**
	 * Is called by @ref Channel to construct a ChannelRead object.
	 *
	 * @note You should never need to use this constructor directly!
	 */
	ChannelRead(ChannelTypePtr channel, Slot* slot) : Base(channel, slot) {}

public:

	/**
	 * Allows to cast @ref ChannelRead and @ref ChannelWrite easily
	 * into a const reference of the wrapped data type.
	 * @throw XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method.
	 */
	operator const ValueType&() const { return this->internalValue(); }

	/**
	 * Returns a const reference on the data.
	 * This is essentially the same as calling internalValue() but simplifies the usage
	 * similar to STL-iterators.
	 * @throw XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method.
	 */
	const ValueType& operator*() const { return this->internalValue(); }

	/**
	 * Returns a const pointer on the data.
	 * This simplifies the usage similar to STL-iterators:
	 *
	 * \code
	 *
	 * stuct MyClass {
	 *     void foo();
	 * }
	 *
	 * ChannelRead<MyClass> data = ...;
	 *
	 * data->foo(); // directly calls foo() of MyClass
	 *
	 * \endcode
	 *
	 * @throw XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method
	 */
	const ValueType* operator->() const { return &this->internalValue(); }

public:
	// make read-methods public

	const Buffer<uint8>& readSerializedValue() {
		return Base::readSerializedValue();
	}

	Buffer<uint8> readSerializedValue(std::list<BinarySerializerCodecPtr>& codecs) {
		return Base::readSerializedValue(codecs);
	}

	void readJSON(JSONValue& oValue) {
		Base::readJSON(oValue);
	}

public:

	/**
	 * Releases the lock explicitly.
	 * @note After calling this method accessing the data results
	 * in an XRuntime exception.
	 */
	void finish() {
		if(this->isValid())
			finish(this->shared.get());
	}

public:

	/**
	 * Returns a read-only channel proxy object of the underlying channel.
	 */
	Channel<T> getChannel();

protected:

	friend class ChannelReadWriteShared<ChannelRead<T> >;

	// API for ~ChannelReadWriteShared
	static void finish(Shared* shared) {
		assert(shared!=NULL);
		if(shared->lock.owns_lock())
			shared->lock.unlock();
	}

	static void discard(Shared* shared) {
		// for read objects a "discard" is equal to a "finish"
		finish(shared);
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * An object that allows exclusive write access to data of a channel.
 * Use @ref Channel::write() to obtain access to data of a channel.
 *
 * The ChannelWrite automatically take care that the data is unlocked if the
 * ChannelWrite object is no longer used and went out of scope. To
 * explicitly release the data by calling the @ref finish() method.
 *
 * After the ChannelWrite released the lock to the data it automatically
 * informs the channel that new data is available. The channel itself
 * will then inform all subscribers of that channel.
 *
 * You can freely copy and reassign the ChannelWrite object to other
 * ChannelWrite variables. The ChannelWrite contains an internal reference
 * counting mechanism. The lock is freed only after all copies have
 * been destroyed or one copy called the @ref finish() method.
 */
template<typename T>
class ChannelWrite : public ChannelReadWriteBase<ChannelWrite<T>, T>
{
protected:
	typedef ChannelReadWriteBase<ChannelWrite<T>, T> Base;

	// push down some important types from our base
	typedef typename Base::Shared     Shared;
	typedef typename Base::ChannelTypePtr ChannelTypePtr;
	typedef typename Base::Slot       Slot;
	typedef typename Base::ValueType  ValueType;


public:

	/**
	 * Default constructor that constructs a ChannelWrite object that
	 * is invalid at the beginning and contains no data. Before you
	 * can use the object you need to assign a valid ChannelWrite
	 * object that was obtained using @ref Channel::write().
	 *
	 * You can use the isValid() method to check if a ChannelWrite object
	 * is valid and contains locked data.
	 */
	ChannelWrite() {}

	/**
	 * Is called by @ref Channel to construct a valid ChannelWrite object
	 * with the corresponding data.
	 *
	 * @note You should never need to use this constructor directly!
	 */
	ChannelWrite(ChannelTypePtr channel, Slot* slot) : Base(channel, slot) {
		this->internalValue().timestamp = Time::now();
	}

public:

	/**
	 * Allows to cast @ref ChannelRead and @ref ChannelWrite easily
	 * into a reference of the wrapped data type.
	 * @throw XRuntime, if the lock on the data was lost,
	 *        e.g. by calling the finish() method
	 */
	operator ValueType&() { return this->internalValue(); }

	/**
	 * Returns a reference on the data.
	 * This is essentially the same as calling internalValue() but simplifies the usage
	 * similar to STL-iterators.
	 * @throw XRuntime, if the lock on the data was lost, 
	 *        e.g. by calling the finish() method
	 */
	ValueType& operator*() { return this->internalValue(); }


	/*const ValueType& operator=(const T& value) {
		this->internalValue() = value;
		return this->internalValue();
	}*/

	const ValueType& operator=(const ValueType& value) {
		this->internalValue() = value;
		return this->internalValue();
	}


	/**
	 * Returns a pointer on the data.
	 * This simplifies the usage similar to STL-iterators:
	 *
	 * \code
	 *
	 * stuct MyClass {
	 *     void foo();
	 * }
	 *
	 * ChannelRead<MyClass> data = ...;
	 *
	 * data->foo(); // directly calls foo() of MyClass
	 *
	 * \endcode
	 *
	 * @throw XRuntime, if the lock on the data was lost, 
	 *        e.g. by calling the finish() method
	 */
	ValueType* operator->() { return &this->internalValue(); }

public:
	// make write-methods public

	void writeSerializedValue(Buffer<uint8> data) {
		Base::writeSerializedValue(std::move(data));
	}

	void writeJSON(const JSONValue& value) {
		Base::writeJSON(value);
	}

public:

	/**
	 * Releases the lock explicitly and informs the Channel to signal all
	 * Subscribers that new data is available.
	 * @note After calling this method accessing the data results
	 * in an XRuntime exception.
	 */
	void finish()
	{
		if(this->isValid())
			finish(this->shared.get());
	}

	/**
	 * Releases the lock explicitly WITHOUT informing the Channel
	 * and without signaling the subscribers. Hence the content that
	 * was written will be ignored and discarded.
	 * You can use this method, e.g. if you notice an error while writing
	 * into the WriteChannel data and want to abort the whole writing
	 * process.
	 */
	void discard()
	{
		if(this->isValid())
			discard(this->shared.get());
	}


public:

	/**
	 * Returns a write-only channel proxy object of the underlying channel.
	 */
	Channel<T> getChannel();

protected:
	friend class ChannelReadWriteShared<ChannelWrite<T> >;

	// API for ~ChannelReadWriteShared
	static void finish(Shared* shared)
	{
		assert(shared!=NULL);
		if(shared->lock.owns_lock()) {
			// release lock object, since it is unlocked by finishWrite()
			shared->lock.release();
			// inform the channel, that the writing was finished
			shared->channel->finishWrite(shared->slot); 
		}
	}

	static void discard(Shared* shared)
	{
		assert(shared!=NULL);
		if(shared->lock.owns_lock()) {
			// release lock object, since it is unlocked by discardWrite()
			shared->lock.release();
			// inform the channel, to discard writing
			shared->channel->discardWrite(shared->slot);
		}
	}
};

///////////////////////////////////////////////////////////////////////////////
// typed read and writes

template<typename T>
struct ChannelReadWriteTraits<ChannelRead<T> >
{
	typedef ConcreteChannel<T>  ChannelType;
	typedef typename ChannelBuffer<T>::Slot SlotType;
	typedef typename ChannelBuffer<T>::ValueType ValueType;
	typedef boost::shared_lock<boost::shared_mutex> Lock; // use shared locks for readers
};

template<typename T>
struct ChannelReadWriteTraits<ChannelWrite<T> >
{
	typedef ConcreteChannel<T>  ChannelType;
	typedef typename ChannelBuffer<T>::Slot SlotType;
	typedef typename ChannelBuffer<T>::ValueType ValueType;
	typedef boost::unique_lock<boost::shared_mutex> Lock; // use unique locks for writers
};

///////////////////////////////////////////////////////////////////////////////
// untyped read and writes

template<>
struct ChannelReadWriteTraits<ChannelRead<void> >
{
	typedef AbstractChannel  ChannelType;
	typedef ChannelBuffer<void>::Slot SlotType;
	typedef ChannelBuffer<void>::ValueType ValueType;
	typedef boost::shared_lock<boost::shared_mutex> Lock; // use shared locks for readers
};

template<>
struct ChannelReadWriteTraits<ChannelWrite<void> >
{
	typedef AbstractChannel  ChannelType;
	typedef ChannelBuffer<void>::Slot SlotType;
	typedef ChannelBuffer<void>::ValueType ValueType;
	typedef boost::unique_lock<boost::shared_mutex> Lock; // use unique locks for writers
};

///////////////////////////////////////////////////////////////////////////////

}


#endif
