/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationSpecialCasesTest.C
 *    Test for serialization framework.
 *
 * @author Erik Einhorn
 */

#include <boost/test/unit_test.hpp>

//#include "BasicClasses.h"
#include "CommonTest.h"

#include <serialization/adapters/std/vector>
#include <xml/XMLDomReflect.h>

using namespace std;
using namespace mira;

///////////////////////////////////////////////////////////////////////////////

class Dummy
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("val", val, "");
	}

	int val;
};

//////////////////////////////////////////////////////////////////////////////

// ClassNestedObjectWithGetter uses a getter to
// access a member that is a struct, where object tracking will be disabled
// by the getter. Additionally, a pointer points the object member. The
// getter now returns a different object than the member. Therefore we
// could get problems if object tracking would be used for the nested
// member Small::i. However, since v is not stored using object tracking
// everything is fine. The pointer will be stored using the value and
// loses its binding to Small::i (as expected: Object tracking and
// pointer bindings will get lost, if getters or setters are used)


class Small
{
public:

	Small() {
		i.val = 333;
	}

	// set member to some values
	Small(bool) {
		i.val = 12;
	}

	// returns true, if the values are as requried
	void check() {
		BOOST_CHECK_EQUAL(i.val, 12);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
	}

	Dummy i;
};

//////////////////////////////////////////////////////////////////////////////

// ClassNestedObjectWithGetter is similar to above class but uses
// a setter. This test case reveals that the NO_TRACKING flag needs
// to be inherited to child objects

class ClassNestedObjectWithSetter
{
public:

	ClassNestedObjectWithSetter() : ptr(NULL) //, ptr2(NULL)
	{}
	ClassNestedObjectWithSetter(bool) : v(true)
	{
		ptr = &v.i;
	}

	void check()
	{
		BOOST_REQUIRE(ptr!=NULL);
		BOOST_CHECK_EQUAL(ptr->val, 12);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("v", v, setter(&ClassNestedObjectWithSetter::setV, this), "");
		r.member("ptr", ptr, "");

	}

	void setV(Small v)
	{
	}

private:

	Small v;
	Dummy*  ptr;
	//Small* ptr2;
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( TestClassNestedObjectWithSetter )
{
	testAll<ClassNestedObjectWithSetter>("Special",2);
}

//////////////////////////////////////////////////////////////////////////////

class ClassWithXMLDom
{
public:

	ClassWithXMLDom()
	{}
	ClassWithXMLDom(bool)
	{
		xml.root().add_child("Test1").add_child("Test1a");
		xml.root().add_child("Test2");
	}

	void check()
	{
		BOOST_CHECK_EQUAL(*xml.root().begin(), "Test1");
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("myxml", xml, "");
	}

public:

	XMLDom xml;
};

//////////////////////////////////////////////////////////////////////////////

template<typename Class>
void testXmlOnly(const char* baseName, int num)
{
	string filename = (boost::format("%s%d.xml") % baseName % num).str();

	// serialize test class
	Class c(true);
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("Test", c);
	xml.saveToFile(filename);

	// deserialize test class
	Class c2; // with 0
	XMLDom xml2;
	xml2.loadFromFile(filename);
	XMLDeserializer s2(xml2.croot());
	s2.deserialize("Test", c2);
	c2.check();
}

BOOST_AUTO_TEST_CASE( TestClassWithXMLDom )
{
	testXmlOnly<ClassWithXMLDom>("Special",3);
}

///////////////////////////////////////////////////////////////////////////////


class SimpleClass
{
public:

	SimpleClass() {
		a = 0;
		b = 0.0f;
	}

	struct Complex {
		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("a", a, "");
			r.member("y", b, "");
		}

		int a,b;
	};


	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
		r.member("b", b, "");
		r.member("c", c, "");
		r.member("d", d, "", Complex());
	}

	int a;
	float b;
	std::string c;
	Complex d;
};

BOOST_AUTO_TEST_CASE( AtomicMemberAsAttributes )
{
	{
	XMLDom xml;
	xml.loadFromString(
		"<root>"
		"<value a=\"321\" b=\"1.23\" c=\"Test\"/>"
		"</root>"
	);

	SimpleClass c;
	XMLDeserializer ds(xml.croot());
	ds.deserialize("value", c);

	BOOST_CHECK_EQUAL(c.a,321);
	BOOST_CHECK_CLOSE(c.b,1.23f,0.01f);
	BOOST_CHECK_EQUAL(c.c,"Test");
	}


	{
	XMLDom xml;
	xml.loadFromString(
		"<root>"
		"<value a=\"321\" b=\"1.23\" c=\"Test\" d=\"oups\"/>"
		"</root>"
	);

	SimpleClass c;
	XMLDeserializer ds(xml.croot());
	BOOST_CHECK_THROW(ds.deserialize("value", c), XIO);

	}

	{
	XMLDom xml;
	xml.loadFromString(
		"<root>"
		"<values>"
		"  <item a=\"1\" b=\"1.11\" c=\"Test1\" />"
		"  <item a=\"2\" b=\"2.22\" c=\"Test2\" />"
		"  <item a=\"3\" b=\"3.33\" c=\"Test3\" />"
		"</values>"
		"</root>"
	);

	std::vector<SimpleClass> c;
	XMLDeserializer ds(xml.croot());
	ds.deserialize("values", c);

	BOOST_CHECK_EQUAL(c.size(),3);
	BOOST_CHECK_EQUAL(c[0].a,1);
	BOOST_CHECK_CLOSE(c[0].b,1.11f,0.01f);
	BOOST_CHECK_EQUAL(c[0].c,"Test1");
	BOOST_CHECK_EQUAL(c[1].a,2);
	BOOST_CHECK_CLOSE(c[1].b,2.22f,0.01f);
	BOOST_CHECK_EQUAL(c[1].c,"Test2");
	BOOST_CHECK_EQUAL(c[2].a,3);
	BOOST_CHECK_CLOSE(c[2].b,3.33f,0.01f);
	BOOST_CHECK_EQUAL(c[2].c,"Test3");
	}



}

///////////////////////////////////////////////////////////////////////////////

