/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ParameterConstructionTest.C
 *    $Test correct creation of classes with parameter constructors.$.
 *
 * @author Ronny Stricker
 * @date   2010/10/25
 */

#include <boost/test/unit_test.hpp>
#include <classes/ParameterConstructorClasses.h>

using namespace std;

#include <type_traits>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

//#############################################################################
//##  ParameterConstructor Test
//#############################################################################

BOOST_AUTO_TEST_CASE( ParameterConstructor )
{
	Object* tObject = NULL;
	
	BOOST_CHECK_THROW(
			ClassFactory::newInstance<mira::ParamConst>( ParamConst::CLASS().getIdentifier() ),
			mira::XLogical );
}

//#############################################################################
//##  ParameterConstructor Test2
//#############################################################################

BOOST_AUTO_TEST_CASE( ParameterConstructor2 )
{
	Object* tConstObj = NULL;

	// try to create object without public default constructor
	BOOST_CHECK_THROW( ParamConst::CLASS().newInstance(), XLogical);

	// however the derived class can be constructed using the default constructor
	tConstObj = ParamConstDerived::CLASS().newInstance();
	BOOST_REQUIRE( tConstObj != NULL );
	delete tConstObj;

	// create new instance from class object
	Class const& tClass = Const::CLASS();
	tConstObj = tClass.newInstance(1,6);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( ((Const*)tConstObj)->mInt, 6 );
	delete tConstObj;
	tConstObj = NULL;

	// create new instance with 0 parameters
	tConstObj = ParamConstDerived::CLASS().newInstance(0);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( tConstObj->getClass().getIdentifier(), "mira::ParamConstDerived" );
	delete tConstObj;
	tConstObj = NULL;

	// create new instance with 1 parameter
	tConstObj = Const::CLASS().newInstance(1,5);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( ((Const*)tConstObj)->mInt, 5 );
	delete tConstObj;
	tConstObj = NULL;

	// create new instance with 2 parameters
	tConstObj = Const2::CLASS().newInstance(2,3,4);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( ((Const2*)tConstObj)->mInt, 3 );
	BOOST_CHECK_EQUAL( ((Const2*)tConstObj)->mInt2, 4 );
	delete tConstObj;
	tConstObj = NULL;

	// create new instance with 3 parameters
	tConstObj = Const3::CLASS().newInstance(3,3,4,5);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( ((Const3*)tConstObj)->mInt, 3 );
	BOOST_CHECK_EQUAL( ((Const3*)tConstObj)->mInt2, 4 );
	BOOST_CHECK_EQUAL( ((Const3*)tConstObj)->mInt3, 5 );
	delete tConstObj;
	tConstObj = NULL;

	// create new instance with 4 parameters
	tConstObj = Const4::CLASS().newInstance(4,3,4,5,6);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( ((Const4*)tConstObj)->mInt, 3 );
	BOOST_CHECK_EQUAL( ((Const4*)tConstObj)->mInt2, 4 );
	BOOST_CHECK_EQUAL( ((Const4*)tConstObj)->mInt3, 5 );
	BOOST_CHECK_EQUAL( ((Const4*)tConstObj)->mInt4, 6 );
	delete tConstObj;
	tConstObj = NULL;

	// try to create new instance with 5 parameters -> should throw
	BOOST_CHECK_THROW( tConstObj = Const4::CLASS().newInstance(5,3,4,5,6,7), XLogical );

	// try to create virtual object
	BOOST_CHECK_THROW( tConstObj = ConstAbstract::CLASS().newInstance(2,3,4), XLogical);
	
	// try to create unknown object
	BOOST_CHECK_THROW( tConstObj = ConstAbstract::CLASS().newInstance("Unknown",2,3,4), XLogical);

	// try to create child of object ConstDerived
	tConstObj = ConstAbstract::CLASS().newInstance("subspace::ConstDerived",2,3,4);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( tConstObj->getClass().getIdentifier(), string("subspace::ConstDerived") );
	BOOST_REQUIRE( dynamic_cast<subspace::ConstDerived*>(tConstObj) != NULL );
	BOOST_CHECK_EQUAL( ((subspace::ConstDerived*)tConstObj)->mInt, 3 );
	BOOST_CHECK_EQUAL( ((subspace::ConstDerived*)tConstObj)->mInt2, 4 );

	// try to create child of object ConstDerived
	tConstObj = ClassFactory::newInstance<Object>("subspace::ConstDerived",2,3,4);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( tConstObj->getClass().getIdentifier(), string("subspace::ConstDerived") );
	BOOST_REQUIRE( dynamic_cast<subspace::ConstDerived*>(tConstObj) != NULL );
	BOOST_CHECK_EQUAL( ((subspace::ConstDerived*)tConstObj)->mInt, 3 );
	BOOST_CHECK_EQUAL( ((subspace::ConstDerived*)tConstObj)->mInt2, 4 );


	delete tConstObj;

	// try to create child of object ConstDerived
	tConstObj = ClassFactory::newInstance<Object>("subspace::ConstDerived",2,3,4);
	BOOST_REQUIRE( tConstObj != NULL );
	BOOST_CHECK_EQUAL( tConstObj->getClass().getIdentifier(), string("subspace::ConstDerived") );
	BOOST_REQUIRE( dynamic_cast<subspace::ConstDerived*>(tConstObj) != NULL );
	BOOST_CHECK_EQUAL( ((subspace::ConstDerived*)tConstObj)->mInt, 3 );
	BOOST_CHECK_EQUAL( ((subspace::ConstDerived*)tConstObj)->mInt2, 4 );
	delete tConstObj;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
