/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Transformer.C
 *    Implementation of Transformer.
 *
 * @author Erik Einhorn
 * @date   2010/09/30
 */

#include <transform/Transformer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TransformerBase::TransformerBase()
{
}

TransformerBase::~TransformerBase()
{
	boost::mutex::scoped_lock lock(mMutex);

	// delete all created nodes
	foreach(auto& v, mNodes)
		delete v.second;
}

bool TransformerBase::addLink(AbstractNodePtr child, AbstractNodePtr parent)
{
	boost::mutex::scoped_lock lock(mMutex);

	// make sure that child and parent really belong to our nodes
	assert(mNodes.count(child->mID)!=0);
	assert(mNodes.count(parent->mID)!=0);

	// if both are already connected, do nothing
	if(child->mParent == parent)
		return false;

	// if child is already connected -> unlink it first from its current parent
	if(child->mParent)
		child->mParent->mChildren.remove(child);

	// connect them
	parent->mChildren.push_back(child);
	child->mParent=parent;
	return true;
}

void TransformerBase::removeLink(AbstractNodePtr child, AbstractNodePtr parent)
{
	boost::mutex::scoped_lock lock(mMutex);

	// make sure that child and parent really belong to our nodes
	assert(mNodes.count(child->mID)!=0);
	assert(mNodes.count(parent->mID)!=0);

	// if both are not connected, do nothing
	if(child->mParent != parent)
		return;

	// remove child from parent
	parent->mChildren.remove(child);
	child->mParent=NULL; // remove parent
}

void TransformerBase::addNode(AbstractNodePtr node)
{
	boost::mutex::scoped_lock lock(mMutex);
	// node must be completely new and should not be connected to anyone
	assert(!node->mParent);
	assert(node->mChildren.empty());

	// insert the node and make sure that it did not exist before
	if(mNodes.insert(std::make_pair(node->mID, node)).second==false) {
		MIRA_THROW(XInvalidParameter,
		           "Cannot add the frame node '" << node->mID << "'. "
		           "A frame node with the same name already exists.");
	}
}

TransformerBase::AbstractNodePtr TransformerBase::getNode(const std::string& nodeID)
{
	boost::mutex::scoped_lock lock(mMutex);

	IDToNodeMap::iterator it = mNodes.find(nodeID);
	if(it==mNodes.end())
		return AbstractNodePtr();
	return it->second;
}

const std::list<TransformerBase::AbstractNodePtr> TransformerBase::getRootNodes()
{
	std::list<TransformerBase::AbstractNodePtr> res;

	boost::mutex::scoped_lock lock(mMutex);

	// add all nodes to the list, that have no parent (and hence are root nodes)
	foreach(auto& v, mNodes)
	{
		AbstractNodePtr node = v.second;
		if(node->mParent==NULL)
			res.push_back(node);
	}

	return res;
}

TransformerBase::AbstractNodePtr TransformerBase::collectNodesUpwardsToRoot(AbstractNodePtr node,
                                                                            std::list<AbstractNodePtr>& oNodes)
{
	oNodes.clear();

	// collect all nodes from 'node' to the root of the node
	oNodes.push_back(node);
	for(int i=0; node->mParent; node = node->mParent, ++i)
	{
		if(i>MAX_TREE_DEPTH)
			MIRA_THROW(XTransform, "The transformation tree's depth exceeded "<<
			           MAX_TREE_DEPTH<<". The transformation tree probably "
			           "contains a loop");

		oNodes.push_back(node->mParent);
	}

	// 'node' is now the root node
	return node;
}

void TransformerBase::getTransformChain(AbstractNodePtr target, AbstractNodePtr source,
		                                Chain& oChain)
{
	assert(source!=NULL);
	assert(target!=NULL);

	// if both are the same, we have nothing to do
	if(source==target)
		return;

	boost::mutex::scoped_lock lock(mMutex);

	// in the following steps we are trying to find the "lowest common ancestor"
	// of the nodes 'source' and 'target' within the transform tree

	// get all nodes from source to towards the root
	AbstractNodePtr sourceRoot = collectNodesUpwardsToRoot(source, oChain.inverse);

	// get all nodes from target to towards the root
	AbstractNodePtr targetRoot = collectNodesUpwardsToRoot(target, oChain.forward);

	// if the root nodes of source and target are not the same, then both are
	// not part of the same tree and hence unconnected.
	if(sourceRoot!=targetRoot)
		MIRA_THROW(XTransform,
		           "Cannot find a transformation between the frames '" <<
		           source->mID << "' and '" << target->mID << "' "
		           "since they are unconnected and not part of the same tree");

	// at this point we have a common ancestor, so find the LOWEST common ancestor now

	// as long as our ancestor is the same, cut it off
	while(!oChain.inverse.empty() && !oChain.forward.empty() &&
		   oChain.inverse.back()  ==  oChain.forward.back()) {
		oChain.inverse.pop_back();
		oChain.forward.pop_back();
	}

	// That's it. Others need 4 times more code to do the same shit ... ;)
}

bool TransformerBase::isTransformAvailable(AbstractNodePtr target, AbstractNodePtr source)
{
	try {
		Chain chain;
		// prepare the chain
		getTransformChain(target, source, chain);
		foreach(AbstractNodePtr node, chain.forward)
			// if node does not contain data transform path is not valid
			if (node->isEmpty())
				return false;
		return true;
	} catch(...) {} // If exception was thrown while preparing the chain ignore it
	// No chain available
	return false;
}

std::list<TransformerBase::AbstractNodePtr> TransformerBase::getNodes()
{
	std::list<AbstractNodePtr> ret;

	boost::mutex::scoped_lock lock(mMutex);

	foreach(auto& v, mNodes)
		ret.push_back(v.second);
	return ret;
}

std::list<std::pair<TransformerBase::AbstractNodePtr, TransformerBase::AbstractNodePtr>> TransformerBase::getLinks()
{
	std::list<std::pair<AbstractNodePtr,AbstractNodePtr>> ret;

	boost::mutex::scoped_lock lock(mMutex);

	foreach(auto& v, mNodes)
	{
		AbstractNodePtr parent = v.second;
		foreach(AbstractNodePtr child, parent->mChildren)
			ret.push_back(std::make_pair(parent,child));
	}
	return ret;
}

///////////////////////////////////////////////////////////////////////////////

}
