/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ManifestAgent.C
 *    $Implementation of ManifestAgent.h$.
 *
 * @author Ronny Stricker
 * @date   2011/02/24
 */

#include <factory/ManifestAgent.h>

#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/map>

#include <algorithm>

using namespace std;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

ManifestAgent::ManifestAgent( std::string const& libraryName )
	: mLibraryName( libraryName )
{

}

ManifestAgent::ManifestAgent( string const& libraryName,
                              vector< ClassProxy > const& classes ) :
	mLibraryName( libraryName )
{
	addClasses( classes );
}

void ManifestAgent::addClasses( vector< ClassProxy > const& classes )
{
	foreach( ClassProxy const& tClass, classes ) {
		MIRA_LOG( DEBUG ) << "new Class: " << tClass.getIdentifier();

		VacantClass* tVacantClass =
				new VacantClass( tClass.getIdentifier(),
				                 tClass.getName(),
				                 tClass.getMetaInfo(),
				                 tClass.isAbstract(),
				                 mLibraryName );

		ManifestClassInfo tClassInfo;
		tClassInfo.mClass = boost::shared_ptr<VacantClass>(tVacantClass);

		// write the parents
		foreach( auto tParent, tClass.getDirectParents() ) {
			tClassInfo.mParents.push_back( tParent.first );
		}
		mClasses.push_back( tClassInfo );
	}
}

void ManifestAgent::loadFile( Path const& filePath )
{
	// save the old vectors
	vector< ManifestClassInfo > tOldClasses = mClasses;

	// deserialize class and parents
	XMLDom xml;
	xml.loadFromFile( filePath );
	XMLDeserializer d( xml.croot() );
	d.deserialize( "Manifest", *this );
	MIRA_LOG(DEBUG) << "Loading manifest " << filePath;

	// register new classes
	foreach( ManifestClassInfo& pInfo, mClasses ) {
		assert( pInfo.mClass );
		pInfo.mClass->mLib =
				( filePath.parent_path() / Path(pInfo.mClass->mLib) ).string();
		MIRA_LOG(DEBUG) << "Registering class " << pInfo.mClass->getIdentifier() 
			<< " from lib " << pInfo.mClass->mLib;
		ClassFactory::registerClass( pInfo.mClass );
	}

	// restore old manifests
	foreach( ManifestClassInfo const& pInfo, tOldClasses ) {
		if ( find( mClasses.begin(), mClasses.end(), pInfo ) == mClasses.end() ) {
			mClasses.push_back( const_cast<ManifestClassInfo&>( pInfo ) );
		}
	}
}

void ManifestAgent::saveToFile( Path const& filePath )
{
	XMLDom xml;
	XMLSerializer s( xml );

	// serialize class and parents
	s.serialize( "Manifest", *this );
	xml.saveToFile( filePath );
}

void ManifestAgent::finalize()
{
	// post register all base classes
	foreach( ManifestClassInfo const& pInfo, mClasses ) {
		try {
			ClassFactory::instance().postRegisterBaseClasses(
				pInfo.mClass->getIdentifier(), pInfo.mParents );
		} catch (Exception& ex) {
			MIRA_LOG(ERROR) << "Failed to finalize class registration for "
			                   "class '" << pInfo.mClass->getIdentifier() <<
			                   "': " << ex.message();
		}
	}
	ClassFactory::instance().finalizePostRegister();

	mClasses.clear();
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
