/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SymbolTable.C
 *    Implementation of SymbolTable.h
 *
 * @author Christian Martin
 * @date   2011/08/27
 */

#include <error/private/SymbolTableWin.h>

#include <error/Logging.h>
#include <error/SystemError.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

SymbolTable::SymbolTable() :
	mInitialized(false)
{
	mProcess = GetCurrentProcess();

	///////////////////////////////////////////////////////////////////////////
	// Initialize the symbol handler for the current process

	// We like to get decorated symbol names
	mOldSymOpts = SymSetOptions((SymGetOptions() & ~SYMOPT_UNDNAME));

	if (SymInitialize(mProcess, NULL, FALSE) != TRUE) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		//MIRA_THROW(XSystemCall, "SymInitialize failed: " << tErr.message());
		MIRA_LOG(ERROR) << "SymInitialize failed: " << tErr.message();
		return;
	}

	///////////////////////////////////////////////////////////////////////////
	// Load all modules based on the Toolhelp-API

	HANDLE tSnapshot = CreateToolhelp32Snapshot(TH32CS_SNAPMODULE, GetCurrentProcessId());
	if (tSnapshot == INVALID_HANDLE_VALUE) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		//MIRA_THROW(XSystemCall, "CreateToolhelp32Snapshot failed: " << tErr.message());
		MIRA_LOG(ERROR) << "CreateToolhelp32Snapshot failed: " << tErr.message();
		return;
	}

	MODULEENTRY32 tModuleEntry;
	tModuleEntry.dwSize = sizeof(tModuleEntry);

	BOOL tKeepGoing = Module32First(tSnapshot, &tModuleEntry);
	while (tKeepGoing) {
		LoadModule(tModuleEntry.szExePath, tModuleEntry.szModule, 
		           (DWORD64)tModuleEntry.modBaseAddr, tModuleEntry.modBaseSize);
		tKeepGoing = Module32Next(tSnapshot, &tModuleEntry);
	}
	CloseHandle(tSnapshot);

	///////////////////////////////////////////////////////////////////////////

	mInitialized = true;
}

SymbolTable::~SymbolTable()
{
	SymCleanup(mProcess);
	SymSetOptions(mOldSymOpts);
}

//////////////////////////////////////////////////////////////////////////////

SymbolTable::SourceLocation SymbolTable::sourceLocation(uint64 addr) const
{
	uint32 tLineNumber = 0;
	std::string tFileName;
	if (getFileAndLineFromAddr(addr, tLineNumber, tFileName))
		return(SourceLocation(tFileName, tLineNumber));
	else
		return(SourceLocation());
}

bool SymbolTable::getModuleNameFromAddr(DWORD64 iAddress, std::string& oName) const
{
	DWORD64 tModBaseAddr = SymGetModuleBase64(mProcess, iAddress);
	if (tModBaseAddr == 0) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_LOG(ERROR) << "SymGetModuleBase64 failed: " << tErr.message();
		oName = "<Unknown moduleABC>";
		return(false);
	}

	// The following code is correct but will not work on the most systems,
	// since it only works, if the used version of DbgHelp.dll (typically
	// from the Windows system32 directory) is the same version as the SDK.
	// (See MSDN for SymGetModuleInfo64). Therefore we use our own small
	// cache for the module names

	/*
	IMAGEHLP_MODULE64 tModule;
	memset(&tModule, 0, sizeof(tModule));
	tModule.SizeOfStruct = sizeof(tModule);
	BOOL tRes = SymGetModuleInfo64(mProcess, tModBaseAddr, &tModule);
	if (tRes != TRUE) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_LOG(ERROR) << "SymGetModuleInfo64 failed: " << tErr.message();
		oName = "<Unknown module>";
		return(false);
	}
	oName = string(tModule.LoadedImageName);
	*/

	std::map<DWORD64, std::string>::const_iterator tIter = mModuleNames.find(tModBaseAddr);
	if (tIter == mModuleNames.end()) {
		oName = "<Unknown moduleDEF>";
		return(false);
	}
	oName = tIter->second;

	return(true);
}

#define STACKWALK_MAX_NAMELEN 1024

bool SymbolTable::getSymbolNameFromAddr(DWORD64 iAddress, std::string& oName, int64& oOffset) const
{
	IMAGEHLP_SYMBOL64* tSymbol = 
		(IMAGEHLP_SYMBOL64*)malloc(sizeof(IMAGEHLP_SYMBOL64)+STACKWALK_MAX_NAMELEN);
	memset(tSymbol, 0x00, sizeof(IMAGEHLP_SYMBOL64)+STACKWALK_MAX_NAMELEN);
	tSymbol->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL64);
	tSymbol->MaxNameLength = STACKWALK_MAX_NAMELEN;

	std::string tName;
	DWORD64 tDisplacement = 0;
	BOOL tSymRes = SymGetSymFromAddr64(mProcess, iAddress, &tDisplacement, tSymbol);
	if (tSymRes != TRUE) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_LOG(ERROR) << "SymGetSymFromAddr64 failed: " << tErr.message();
		tName = "<Unknown symbol>";
		tDisplacement = 0;
	} else 
		tName = std::string(tSymbol->Name);

	free(tSymbol);

	oName = tName;
	oOffset = tDisplacement;

	return(tSymRes == TRUE);
}

bool SymbolTable::getFileAndLineFromAddr(DWORD64 iAddress, uint32& oLine, std::string& oFileName) const
{
	IMAGEHLP_LINE64 tLine;
	memset(&tLine, 0x00, sizeof(tLine));
	tLine.SizeOfStruct = sizeof(tLine);
	DWORD tDisplacement = 0;
	BOOL tRes = SymGetLineFromAddr64(mProcess, iAddress, &tDisplacement, &tLine);
	if (tRes != TRUE) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_LOG(ERROR) << "SymGetLineFromAddr64 failed: " << tErr.message();
		oLine = 0;
		oFileName = "<Unknown file>";
		return(false);
	}

	oLine = tLine.LineNumber;
	oFileName = tLine.FileName;
	return(true);
}

///////////////////////////////////////////////////////////////////////////////

BOOL SymbolTable::LoadModule(LPCSTR iBinaryName, LPCSTR iModuleName, 
                             DWORD64 iDLLBaseAddr, DWORD iDLLSize)
{
	if ((iBinaryName == NULL) || (iModuleName == NULL)) {
		MIRA_LOG(ERROR) << "Invalid binary or module name.";
		return(FALSE);
	}

	DWORD64 tModBaseAddr = SymLoadModule64(mProcess, 0, iBinaryName, iModuleName, iDLLBaseAddr, iDLLSize);
	if (tModBaseAddr == 0) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		//MIRA_THROW(XSystemCall, "SymLoadModule64 failed: " << tErr.message());
		MIRA_LOG(ERROR) << "SymLoadModule64 failed: " << tErr.message();
		return(FALSE);
	}

	// We need this to cache the binary names of the modules, since the
	// function SymGetModuleInfo64 depends on the installed version of
	// DbgHelp.dll.
	mModuleNames[tModBaseAddr] = std::string(iBinaryName);

	//
	// Unused code for retrieving the file-version. This code is used at the moment,
	// but may be need this later. If we use this, we also must add Version.lib
	// in the CMakeLists.error!
	//
	/*
	// Try to retrive the file-version
	DWORD tFHandle;
	DWORD tSize = GetFileVersionInfoSizeA(iBinaryName, &tFHandle);
	if (tSize > 0) {
		LPVOID tVersionData = malloc(tSize);
		if (tVersionData != NULL) {
			if (GetFileVersionInfoA(iBinaryName, tFHandle, tSize, tVersionData) != 0) {
				UINT tLen;
				VS_FIXEDFILEINFO *tFileInfo = NULL;
				if (VerQueryValue(tVersionData, TEXT("\\"), (LPVOID*)&tFileInfo, &tLen) != TRUE) {
					tFileInfo = NULL;
				} else {
					// Get file version number from tFileInfo pointer...
				}
			}
			free(tVersionData);
		}
	}
	*/

    return(TRUE);
}

///////////////////////////////////////////////////////////////////////////////

}
